(ns ulisse.core
  (:refer-clojure :exclude [eval cast])
  (:require [spyscope.core]
            [codox-md.writer :only [write-docs]]
            [codox-md.markdown :only [md]]
            [markdown.core :only [md-to-html-string]])
  (:import [org.python.util PythonInterpreter]
           [org.python.core PyObject PyString]
           [poppea]
           [clojure.string]))

(defn highlighter []
  (doto (PythonInterpreter.)
    (.exec
"import pygments
import pygments.lexers
import pygments.formatters")))

(defn cast [^PyObject obj class]
  (.__tojava__ obj class))

(defn eval
  ([^PythonInterpreter interpreter function]
     (.eval interpreter function))
  ([^PythonInterpreter interpreter function class]
     (cast (.eval interpreter function) class)))

(defmulti to-jython class)

(defmethod to-jython String [obj] (PyString. obj))
(defmethod to-jython PyObject [obj] obj)

(defn call [^PyObject callable & args]
  (if (empty? args)
    (.__call__ callable)
    (.__call__ callable (->> args
                             (map to-jython)
                             (into-array PyObject)))))

(defn call2 [^PythonInterpreter interpreter name & args]
  (apply call (eval interpreter name) args))

(defn highlight
  ([highlighter code]
     (highlight highlighter code nil))
  ([^PythonInterpreter highlighter code lexer]
     (let [html (call2 highlighter "pygments.formatters.HtmlFormatter")
           l (if lexer
               (call2 highlighter (str "pygments.lexers." lexer))
               (call2 highlighter "pygments.lexers.guess_lexer" code))]
       (-> highlighter
           (call2 "pygments.highlight" code l html)
           (cast java.lang.String)))))

(defn write-docs
  "Take raw documentation and turn it into formatted HTML."
  [project]
  (with-redefs [codox.markdown/md markdown.core/md-to-html-string]
    #spy/p "HELLO"
    (codox.writer/write-docs project)))

(defn test2 []
  (println (highlight (highlighter) "print(\"hello world!\"")))
