(ns net.cgrand.xforms.pv
  (:refer-clojure :except [pop peek])
  (:require [clojure.core :as clj]
    [clojure.test :as test :refer [deftest testing is are]]))

(defn node-push [node items]
  (if-some [{:keys [p tail rot] :or {rot 0}} node]
    (let [ring (clj/peek tail)
          unring (into (subvec ring 0 (- 32 rot)) (subvec items 0 rot))
          ring (into (subvec items rot 32) (subvec ring (- 32 rot) 32))
          tail (conj (clj/pop tail) unring)]
      (if (= 32 (count tail))
        {:p (node-push p tail) :tail [ring] :rot rot}
        {:p p :tail (conj tail ring) :rot rot}))
    {:tail [items]}))

(defn push [pv x]
  (if-some [{:keys [p tail]} pv]
    (if (= 32 (count tail))
      {:p (node-push p tail) :tail [x]}
      {:p p :tail (conj tail x)})
    {:tail [x]}))

(deftest push-rot
  (is (= (push {:p {:tail [(into (vec (range 29)) '[A B C])] :rot 3} :tail (vec (range 32 64))} '$)
        {:p {:p nil :tail [(into (vec (range 29)) (vec (range 32 35))) (into (vec (range 35 64)) '[A B C])] :rot 3} :tail '[$] })))

(defn peek [{:keys [p tail]}]
  (clj/peek tail))

(defn node-pop [node]
  (when-some [{:keys [p tail rot]} node]
    (let [ring (clj/peek tail)]
      (if-some [[p tail] (if (= 1 (count tail)) (node-pop p) [p (clj/pop tail)])]
        (let [last (clj/peek tail)
              unring (into (subvec last (- 32 rot) 32) (subvec ring 0 (- 32 rot)))
              ring (into (subvec last 0 (- 32 rot)) (subvec ring (- 32 rot) 32))]
          [{:p p :tail (conj (clj/pop tail) ring) :rot rot} unring])
        (let [unring (into (subvec ring (- 32 rot) 32) (subvec ring 0 (- 32 rot)))]
          [nil unring])))))

(defn pop [pv]
  (if-some [{:keys [p tail]} pv]
    (if (= 1 (count tail))
      (when-some [[p tail] (node-pop p)]
        {:p p :tail tail})
      {:p p :tail (clj/pop tail)})
    (throw (RuntimeException. "Can't pop on empty."))))

(defn shiftr [node x]
  (if-some [{:keys [p tail rot] :or {rot 0}} node]
    (let [ring (clj/peek tail)
          y (nth ring (- 31 rot))
          ring (assoc ring (- 31 rot) x)
          rot' (bit-and (inc rot) 31)]
      (if (zero? rot')
        (let [[p last] (shiftr p ring)]
          [{:p p :tail (into [last] (clj/pop tail)) :rot rot'} y])
        [{:p p :tail (conj (clj/pop tail) ring) :rot rot'} y]))
    [nil x]))

(defn shiftl [node x]
  (if-some [{:keys [p tail rot] :or {rot 0}} node]
    (if (zero? rot)
      (let [[p first] (shiftl p (nth tail 0))
            y (nth first 0)
            ring (into [x] (subvec first 1 32))
            tail (conj (subvec tail 1) ring)]
        [{:p p :tail tail :rot 31} y])
      (let [ring (clj/peek tail)
            rot' (dec rot)
            y (nth ring (- 31 rot'))
            ring (assoc ring (- 31 rot') x)]
        [{:p p :tail (conj (clj/pop tail) ring) :rot rot'} y]))
    [nil x]))

(defn pushl [pv x]
  (if-some [{:keys [p tail]} pv]
    (if (= 32 (count tail))
      (let [[p y] (shiftr (node-push p tail) x)]
        {:p p :tail [y]})
      (let [[p y] (shiftr p x)]
        {:p p :tail (into [y] tail)}))
    {:tail [x]}))

(defn popl [pv]
  (if-some [{:keys [p tail]} pv]
    (let [[p _] (shiftl p (nth tail 0))] 
      (if (= 1 (count tail))
        (let [[p tail] (node-pop p)]
          {:p p :tail tail})
        {:p p :tail (subvec tail 1)}))
    (throw (RuntimeException. "Can't pop on empty."))))

(defn length [pv-or-node]
  (if-some [{:keys [p tail]} pv-or-node]
    (+ (count tail) (* 32 (length p)))
    0))

(defn node-lookup [{:keys [p tail]} idx]
  (let [tail-offset (* 32 (length p))]
    (if (< idx tail-offset)
      (let [idx (- idx (:rot p 0))
            idx (if (neg? idx) (+ idx tail-offset) idx)
            v (node-lookup p (bit-shift-right idx 5))]
        (nth v (bit-and 31 idx)))
      (nth tail (- idx tail-offset)))))

(defn lookup [pv idx]
  (if (and (<= 0 idx) (< idx (length pv)))
    (node-lookup pv idx)
    (throw (IndexOutOfBoundsException.))))

(defn node-cat [a b]
  (cond
    (nil? a) b
    (nil? b) a
    :else ; a and b rotations may differ
    ; if a is rotated by ra it means there's ra items at the back that are logically at the front
    ; same goes for rb
    ; 
    (let [{pa :p ta :tail} pva
          {pb :p tail :tail} (reduce pushl pvb (rseq ta))]
      {:p (cat pa pb) :tail tail})))

(defn cat [pva pvb]
  (cond
    (nil? pva) pvb
    (nil? pvb) pva
    :else
    (let [{pa :p ta :tail} pva
          {pb :p tail :tail} (reduce pushl pvb (rseq ta))]
      {:p (cat pa pb) :tail tail})))