;  Copyright (C) 2020 Gabriel Ash

; This program and the accompanying materials are made available under
; the terms of the Eclipse Public License 2.0 which is available at
; http://www.eclipse.org/legal/epl-2.0 .
 
; This code is provided as is, without any guarantee whatsoever.

  (ns antifessional.libmisc.printf
    "printf variants (formatted output)
    
    ![](images/antifessional.libmisc.printf.png)
    "
    (:require 
     [clojure.string :as str]
     [clojure.java.io :as java.io]     

     [gabrielash.defnw.core :refer :all]
     
     [antifessional.libmisc.core :refer :all]
     [antifessional.libmisc.args :refer :all] 
     [antifessional.libmisc.string :refer [in?]]))
 




;;;;;;;;;;;;;;;;;;;;;;;;;; 
;;;;;;;;; SPRINTF
;;;;;;;;;;;;;;;;;;;;;;;;;; 

(defn- ->int [x]
  {:pre [(number? x)]}

  (cond (instance? clojure.lang.BigInt x)  (biginteger x)
        :else x))

(defn- ->dbl [x]
  {:pre [(number? x)]}
  
  (cond 
       (instance? clojure.lang.BigInt x) (double x)
        (instance? clojure.lang.Ratio x) (double x)
        (int? x) (double x)
        :else x))


(defn- ->str [x]
  (if (nil? x)
    "|NIL|"
    x))

(defn- ->ratio [x]
  {:pre [(instance? clojure.lang.Ratio x)]}
  x) 


;;;;;;;;;;;;;;;;;;;;

(defnw- cast-argv
  "casts a vector of format arguments to the correct type
   according to the format string"
  [fmt-str cast-dispatch argv]

  {:let [conversions
         (->> fmt-str
              (re-seq #"%[^a-zA-Z%]*([a-zA-Z%])")
              (map (comp str/lower-case last))
              (remove (partial in? "n%")))]

   :cond [(!= (count conversions) (count argv))
          (throw (ex-info
                  (str "format string <-> arguments mismatch: expected "
                       (count conversions) " args but received " (count argv))
                  {:argv argv :conversions conversions}))]}

  (let [cast-fn (fn [type arg]
                  ((get cast-dispatch type identity) arg))]

    (into []
          (map cast-fn conversions argv))))



(defn sprintf 
  "like [format](https://clojuredocs.org/clojure.core/format), 
   but casts numeric arguments to a type that matches
   the format conversion:
   
   *  both floating point and integer conversions handle BigIntegers.
   *  floating point conversions handle ratios.
   *  special formatter %r handles ratios
   *  %s handles nil  (rendered as `|NIL|`).
   *  %s accepts all types"
  [fmt-string & argv]
  (let [tag->cast-fn {"d" ->int
                      "x" ->int
                      "o" ->int
                      "f" ->dbl
                      "g" ->dbl
                      "e" ->dbl
                      "a" ->dbl
                      "r" ->ratio
                      "s" ->str}]
    (apply format
           (str/replace fmt-string #"%r" "%s")
           (cast-argv fmt-string
                   tag->cast-fn
                   argv))))


(defn errf
  "Like [printf](https://clojuredocs.org/clojure.core/printf),
   But prints to *err*.
   The the formar string uses [](sprintf)."
  [destination fmt-string & argv]
  (binding [*err* destination]
    (print (apply sprintf fmt-string argv))))


(defn printfln
  "like [printf](https://clojuredocs.org/clojure.core/printf), 
   but casts arguments to a type that matches
   the format conversion (uses [[sprintf]]) and adds a newline"
  [fmt & argv]
  (println
   (apply sprintf
          fmt
          argv)))
