(ns antifessional.libmisc.io
  "filesystem operations
  
    ![](images/antifessional.libmisc.io.png)
  "
  (:require

   [clojure.java.io :as cio]
   [clojure.edn :as edn]
   [clojure.string :as str]

   [antifessional.libmisc.core :refer :all]
   [antifessional.libmisc.ex :refer [throw-ex]]

   ))


(defn slurp-edn
  "Slurp edn from an io/reader source (filename or io/resource)."
  [source]
  (try
    (with-open [r (cio/reader source)]
      (edn/read (java.io.PushbackReader. r)))

    (catch java.io.IOException e
      (throw-ex "Cannot open '%s': %s\n" source (.getMessage e) {:cause e}))
    (catch RuntimeException e
      (throw-ex
                    "Error parsing edn file '%s': %s\n" source (.getMessage e) {:cause e}))))


(defn zap-directory
  "Recursively delete a directory."
  [file-name]
  (let [file (cio/file file-name)]
    (when (.isDirectory file)
      (doseq [file-in-dir (.listFiles file)]
        (zap-directory file-in-dir)))
    (cio/delete-file file)))

(defn files
  "Returns lazy seq of files in `dir` (recursively).
   If a `regex` is given as second argument, filters by regex and returns a lazy 
   seq of file names.
   :only-files true -> removes directories from result"
   
  ([dir]
   (file-seq (cio/file dir)))

  ([dir regex & {:keys [only-files]}]
   (->> dir
        cio/file
        file-seq
        ((fn [fs] (if only-files
                    (filter #(.isFile %) fs)
                    fs)))
        (map str)
        (filter #(re-find regex %)))))        


(defn cp!
  "Copies file-path to output-path.
  If output file exists, clobber if `:clobber` is true, otherwise throws ExceptionInfo."
  [file-path output-path & {:keys [clobber]}]
  (let [out (cio/file output-path)
        in (cio/file file-path)]
    (when (.exists out)
      (if clobber
        (cio/delete-file out)
        (throw-ex "Temp file %s already exists!" output-path)))
    (cio/copy in out)))



(defn resource-prefix!
  "get the correct prefix (absolute) when using the resource in dev.
   Must supply the relative path of the library directory. "
  [resource-segment]
  (if-let [prefix (some-> resource-segment
                          cio/resource
                          .getFile
                          (str/replace #"%20" " "))]
    (if (str/ends-with? prefix "/")
      prefix
      (str prefix "/"))
    (throw-ex "Something is wrong with the resource tree (dev!)" )))
