;  Copyright (C) 2020 Gabriel Ash
  
; This program and the accompanying materials are made available under 
; the terms of the Eclipse Public License 2.0 which is available at
; http://www.eclipse.org/legal/epl-2.0 .
 
; This code is provided as is, without any guarantee whatsoever.

(ns antifessional.libmisc.core
  "aliases, alternatives, and enhancements to core functions

   
  |  name             | args                     |  action                                      |
  |  ---------------- | -----------------------  |  ------------------------------------------- |
  |  *0               |                          |  VAR: last value captured by #np             |
  |  none-fn          |  f & fs                  |  f->true if all fs return false              |
  |  some*            |  pred coll               |  -> first elt of coll : (pred coll)->true    |
  |  atom?            |  x                       |  true if x is atom                           |
  |  regex?           |  x                       |  true if x is regex                          |
  |  !=               |  x & xs                  |  alias not=                                  |
  |  !                |  x                       |  alias not                                   |
  |  !!               |  f & args                |  macro : (not (f args))                      |
  |  is=              |  x                       |  alias of (partial = x)                      |
  |  assoc-if         |  col  test k v           |  macro : assoc if test (short-circuits)      |
  |  assoc-iff        |  col  test k v           |  assoc if test       (function)              |
  |  assoc*           |  col  k v                |  assoc unless nil? v                         |
  |  pt>              |  f & args                |  alias partial                               |
  |  pt<              |  f & args                |  partial with new arguments inserted first   |
  |  count=1          |  col                     |  (= 1 (count coll))                          |
  |  strip-nils       |  map                     |  removes nil values from flat map            |
  |  no-nil-vals?     |  col                     |  true if no nil value at any depth           |
  |  flatten-any      |  col                     |  flatten but works for maps and sets as well |
  |  flatten*         |  seq                     |  flatten but returns nil on empty            |
  |  has-nils?        |  col                     |  true if no nil keys or vals at any depth    |
  |  fails?           |  expr ?:out ?:error      |  true if expr throws :error                  |
  |  failsv?          |  expr                    |  true if expr throws and prints error data   |

  ![](images/antifessional.libmisc.core.png)
  "
  (:require [antifessional.libmisc.np]
            [potemkin :refer [import-vars]]))


(import-vars [antifessional.libmisc.np *0])

#_(set! *data-readers* (assoc *data-readers* 'np #'antifessional.libmisc.core/np*))
#_(set! *data-readers* (assoc *data-readers* 'np (fn [f] `(fn [& args#]
                                                            (let [v# (apply ~f args#)]
                                                              (alter-var-root #'*0
                                                                              (constantly v#)))
                                                            nil))))


(defn none-fn
  "F1...Fn -> (and (not F1 )...(not Fn))   "
  [f & fs]
  (complement (apply some-fn f fs)))

(defn some*
  "replaces [some](https://clojuredocs.org/clojure.core/some) 
   : returns the first true **value** rather than `true`"
  [pred coll]
  (some #(and (pred %) %) coll))

(defn atom?
  "is it an atom?  *(java wrapper)*"
  [a]
  (instance? clojure.lang.Atom a))

(defn regex?
  "is it a regex?  *(java wrapper)*"
  [rx]
  (instance? java.util.regex.Pattern rx))



(def !=
  "alias of [not=](https://clojuredocs.org/clojure.core/not=) "

  #'not=)

(def !
  "alias of [not](https://clojuredocs.org/clojure.core/not) "

  #'not)

(defmacro !!
  "applies [complement](https://clojuredocs.org/clojure.core/complement) of `f` to
  `args`"
  [f & args]

  `(apply (complement ~f) [~@args]))

(defn is=?
  "returns a predicate that tests for equality with its `arg` "
  [arg] #(= arg %))


(defn ^:forward no-nils-? [col])

(defn- no-nils--?
  [col]

  (and (cond (sequential? col) (!! seq (filter nil? col))
             (set? col)        (= (count col) (count (disj col nil)))
             (map? col)       (!! seq (keep #(when (nil? (val %))
                                               (key %)) col)))

       (every? no-nils-?
               (if (map? col) (vals col) col))))

(defn- no-nils-?
  "true only if v includes no nil values at any depth, nil -> false"
  [v]
  (if (coll? v)
    (no-nils--? v)
    (!! nil? v)))

(defn no-nil-vals?
  "true only if `col` includes no `nil` values at any depth,  
   `nil` -> `nil`  
   intended for testing and sanity checks"

  [col]

  {:pre [((some-fn sequential? nil? map? set?) col)]}
  (if (nil? col)
    nil
    (no-nils-? col)))

(defn strip-nils
  "removes `nil` values from `map` (not recursive)"
  [map]
  (apply dissoc map
         (keep #(when (nil? (val %))
                  (key %)) map)))

(defn flatten-any
  "like [flatten](https://clojuredocs.org/clojure.core/flatten)
  but works on maps and sets as well"
  [x]
  {:pre [((some-fn coll? nil?) x)]}
  (if (nil? x)
    nil
    (remove coll?
            (rest (tree-seq coll? seq x)))))

(defn flatten*
  "replaces [flatten](https://clojuredocs.org/clojure.core/flatten)
   but `nil` -> `nil`"
  [x]
  {:pre [((some-fn seq? nil?) x)]}
  (if (nil? x)
    nil
    (remove seq?
            (rest (tree-seq sequential? seq x)))))

(defn has-nils?
  "find `nil` anywhere in a deep collection, including in keys"
  [col]
  (some nil? (flatten-any col)))

(defn- partial<
  "reverse [partial](https://clojuredocs.org/clojure.core/partial)  "
  [f & argv]
  (fn [& args]
    (apply f (concat args argv))))

(def pt>
  "alias for [partial](https://clojuredocs.org/clojure.core/partial)"
  #'partial)

(def pt<
  "alias for [[partial<]]"
  #'partial<)

(defn assoc*
  "like [assoc](https://clojuredocs.org/clojure.core/assoc)
   but ignores `nil` values.  
   cannot be used on vectors"
  ([col k v]
   (if-not (nil? v)
     (assoc col k v)
     col))

  ([col k v & kvs]
   (merge col
          (strip-nils
           (apply hash-map
                  (concat [k v] kvs))))))

(defn assoc-iff
  "conditional [assoc](https://clojuredocs.org/clojure.core/assoc)
   "
  ([col test k v]
   (if test (assoc col k v) col)))

(defmacro assoc-if
  "like [[assoc-iff]] but short-circuiting macro"
  [coll test k v]

  `(let [coll# ~coll]
     (if ~test (assoc coll# ~k ~v)
         coll#)))


(defn count=1?
  "convenience function to test for single element collections"
  [x]
  (= 1 (count x)))


(defmacro fails?
  "true if `expr` throws java.lang.Exception; false otherwise;
   :out true -> prints type and exception
   :error ExceptionClass -> catches only specific class"
  [expr & {:keys [out error] :or {error java.lang.Exception}}]
  `(try ~expr false
        (catch ~error e#
          (when ~out
            (println "FAILS! with " (class e#) "\n" (.getMessage e#)))
          true)))

(defmacro failsv?
  "Verbose wrapper of [[fails?]] that prints error type and message"
  [expr]
  `(fails? ~expr :out true))

nil
