(ns leiningen.new.nedap-lib
  "Generate a library project."
  (:require
   [clojure.string :as string]
   [clojure.java.io :as io]
   [leiningen.new.templates :refer [renderer year date project-name ->files sanitize-ns name-to-path multi-segment]]
   [leiningen.core.main :as main])
  (:import
   (java.io File Reader)
   (java.util Properties)))

(def properties-map
  (delay
   (let [filename "nedap.lein-template.properties"]
     (when (-> filename File. .exists)
       (with-open [^Reader reader (io/reader filename)]
         (doto (Properties.)
           (.load reader)))))))

(defn get-system-property [k]
  (or (get @properties-map k)
      (System/getProperty k)))

(defn gather [{:keys [prompt iteration-prompt parse-fn]}]
  (println prompt)
  (loop []
    (some-> iteration-prompt println)
    (if-some [choice (parse-fn (read-line))]
      (do
        (println)
        choice)
      (recur))))

(defn ask [prompt default]
  (gather {:prompt   (str prompt
                          (when default
                            (str "\n(default: " default ")")))
           :parse-fn (fn [v]
                       (cond
                         (string/blank? v) (do
                                             (when-not default
                                               (println "Please enter a value."))
                                             default)
                         true              v))}))

(defn confirmation [prompt]
  (gather {:prompt           prompt
           :iteration-prompt "(respond with 'yes' or 'no')"
           :parse-fn         (fn [v]
                               (cond
                                 (#{"y" "yes" "Y" "YES" "Yes"} v) true
                                 (#{"n" "no" "N" "NO" "No"} v)    false))}))

(defn env-truth [k]
  (let [truthy (fn [x]
                 (not (-> #{nil
                            'null
                            ""
                            false
                            'n
                            'no
                            'off
                            0}
                          (contains? x))))]
    (some-> k get-system-property .toLowerCase (string/replace #"( |'|\")" "") read-string truthy)))

(defn nedap-lib [raw-name]
  (let [default-github-username "nedap"
        github-username (or (get-system-property "nedap.lein-template.github_username")
                            (ask "Github user/organisation that will own the repo:"
                                 default-github-username))
        nedap? (= github-username
                  default-github-username)
        copyright-holder (or (get-system-property "nedap.lein-template.copyright_holder")
                             (ask "Copyright holder (example: My Company):"
                                  (when nedap?
                                    "Nedap")))
        group-id (or (get-system-property "nedap.lein-template.group_id")
                     (ask "Maven group id (example: com.mycompany):"
                          (when nedap?
                            "com.nedap.staffing-solutions")))
        prefix (or (get-system-property "nedap.lein-template.lib_prefix")
                   (ask "Clojure namespace prefix:"
                        (if nedap?
                          (str "nedap." raw-name)
                          (str group-id "." raw-name))))
        gpg-key (or (get-system-property "nedap.lein-template.gpg_email")
                    (if nedap?
                      "releases-staffingsolutions@nedap.com"
                      (ask "Email for GPG-signing releases:"
                           nil)))
        nvd? (if-some [choice (env-truth "nedap.lein-template.use_nvd")]
               choice
               (confirmation "Setup lein-nvd?"))
        clojurescript? (if-some [choice (env-truth "nedap.lein-template.use_cljs")]
                         choice
                         (confirmation "Also setup ClojureScript?"))
        web? (if-some [choice (env-truth "nedap.lein-template.use_pedestal")]
               choice
               (confirmation "Setup a Pedestal web application?"))
        ssr? (if-not web?
               false
               (if-some [choice (env-truth "nedap.lein-template.use_ssr")]
                 choice
                 (confirmation "Add `assets` and `layout` and  components to the Pedestal app? (These are not needed in API only projects)")))
        system? (or web?
                    (if-some [choice (env-truth "nedap.lein-template.use_system")]
                      choice
                      (confirmation "Setup a com.stuartsierra/component System?")))
        clojars? (if-some [choice (env-truth "nedap.lein-template.use_clojars")]
                   choice
                   (confirmation "Deploy to Clojars?"))
        license? (if-some [choice (env-truth "nedap.lein-template.do_license")]
                   choice
                   (confirmation "Publicly license?"))
        monorepo-subproject? (and (-> "../.git" java.io.File. .exists)
                                  (let [f (java.io.File. "../project.clj")]
                                    (if-not (-> f .exists)
                                      true
                                      (not (-> f slurp (clojure.string/starts-with? "(defproject nedap-lib/lein-template"))))))
        render (renderer "nedap-lib")
        escaped-prefix (string/replace prefix "." "\\.")
        main-ns (str prefix ".api")
        system-ns (str prefix ".system")
        web-source-paths (if web?
                           " \"modules\" \"libs\""
                           "")
        data {:clojars?             clojars?
              :clojurescript?       clojurescript?
              :copyright-holder     copyright-holder
              :date                 (date)
              :escaped-prefix       escaped-prefix
              :dev-source-paths     (str "\"src\" \"dev\" \"test\""
                                         web-source-paths)
              :web-source-paths     web-source-paths
              :github-username      github-username
              :gpg-key              gpg-key
              :group-id             group-id
              :initial-release      "unreleased"
              :license?             license?
              :monorepo-subproject? monorepo-subproject?
              :name                 (project-name raw-name)
              :namespace            main-ns
              :nvd?                 nvd?
              :system-ns            system-ns
              :nedap?               nedap?
              :system?              system?
              :system-path          (name-to-path system-ns)
              :ssr?                 ssr?
              :nested-dirs          (name-to-path main-ns)
              :prefix               prefix
              :prefix-dir           (name-to-path prefix)
              :raw-name             raw-name
              :web?                 web?
              :year                 (year)}
        web-files (when web?
                    (->> {"web/config/component.clj"        "config.component"
                          "web/config/getters.clj"          "config.getters"
                          "web/config/pedestal_adapter.clj" "config.pedestal-adapter"
                          "web/config/kws.clj"              "config.kws"
                          "web/config/protocols/config.clj" "config.protocols.config"
                          "web/environment/kws.clj"         "environment.kws"
                          "web/router/component.clj"        "router.component"
                          "web/router/kws.clj"              "router.kws"}
                         (mapv (fn [[k v]]
                                 [(str "modules/"
                                       (name-to-path (str prefix "." v))
                                       ".clj")
                                  (render k data)]))))
        ssr-files (when ssr?
                    (->> {"web/assets/component.clj"        "assets.component"
                          "web/assets/kws.clj"              "assets.kws"
                          "web/assets/protocols/assets.clj" "assets.protocols.assets"
                          "web/layout/api.clj"              "layout.api"
                          "web/layout/component.clj"        "layout.component"
                          "web/layout/kws.clj"              "layout.kws"
                          "web/layout/protocols/layout.clj" "layout.protocols.layout"
                          "web/layout/views/main.clj"       "layout.views.main"}
                         (mapv (fn [[k v]]
                                 [(str "modules/"
                                       (name-to-path (str prefix "." v))
                                       ".clj")
                                  (render k data)]))))]
    (cond->> [["nedap.lein-template.properties" (render "nedap.lein-template.properties" data)]
              ["project.clj" (render "project.mustache" data)]
              ["README.md" (render "README.md" data)]
              [".gitignore" (render "gitignore" data)]
              (when nvd?
                ["nvd_suppressions.xml" (render "nvd_suppressions.xml" data)])
              (when web?
                ["resources/logback.xml" (render "logback.xml" data)])
              (when web?
                ["resources/config.edn" (render "config.edn" data)])
              (when ssr?
                [(str "src/" (name-to-path (str prefix ".stylesheet")) ".clj")
                 (render "web/stylesheet.clj" data)])
              [".github/pull_request_template.md" (render "pull_request_template.md" data)]
              [".github/PULL_REQUEST_TEMPLATE/ncrw.md" (render "ncrw_pull_request_template.md" data)]
              [".github/ISSUE_TEMPLATE/bug.md" (render "bug.md" data)]
              [".github/ISSUE_TEMPLATE/feature.md" (render "feature.md" data)]
              [".github/ISSUE_TEMPLATE/technical_improvement.md" (render "technical_improvement.md" data)]
              ["dev/dev.clj" (render "dev.clj" data)]
              (when system?
                [(str "src/{{system-path}}.clj" (when clojurescript?
                                                  "c"))
                 (render "system.clj" data)])
              (when-not web?
                [(str "src/{{nested-dirs}}.clj" (when clojurescript?
                                                  "c"))
                 (render "api.cljc" data)])
              (when-not web?
                [(str "test/unit/{{nested-dirs}}.clj" (when clojurescript?
                                                        "c"))
                 (render "test.cljc" data)])
              (when system?
                ["test/{{prefix-dir}}/global_test_setup.clj" (render "global_test_setup.clj" data)])
              (when clojurescript?
                ["test/{{prefix-dir}}/test_runner.cljs" (render "test_runner.cljs" data)])
              (when license?
                ["LICENSE" (render "LICENSE" data)])
              [".circleci/config.yml" (render "config_yml.mustache" data)]
              "resources"]
      web?              (into web-files)
      ssr?              (into ssr-files)
      (and web? nedap?) (into [[(str "modules/"
                                     (name-to-path (str prefix ".config.legacy-adapter"))
                                     ".clj")
                                (render "web/config/legacy_adapter.clj" data)]])
      true              (filter some?)
      true              (apply ->files data))))
