(ns leiningen.new.nedap-lib
  "Generate a library project."
  (:require
   [clojure.string :as string]
   [leiningen.new.templates :refer [renderer year date project-name ->files sanitize-ns name-to-path multi-segment]]
   [leiningen.core.main :as main]))

(defn gather [{:keys [prompt iteration-prompt parse-fn]}]
  (println prompt)
  (loop []
    (some-> iteration-prompt println)
    (if-some [choice (parse-fn (read-line))]
      (do
        (println)
        choice)
      (recur))))

(defn ask [prompt default]
  (gather {:prompt   (str prompt
                          (when default
                            (str "\n(default: " default ")")))
           :parse-fn (fn [v]
                       (cond
                         (string/blank? v) (do
                                             (when-not default
                                               (println "Please enter a value."))
                                             default)
                         true              v))}))

(defn confirmation [prompt]
  (gather {:prompt           prompt
           :iteration-prompt "(respond with 'yes' or 'no')"
           :parse-fn         (fn [v]
                               (cond
                                 (#{"y" "yes" "Y" "YES" "Yes"} v) true
                                 (#{"n" "no" "N" "NO" "No"} v)    false))}))

(defn env-truth [k]
  (let [truthy (fn [x]
                 (not (-> #{nil
                            'null
                            ""
                            false
                            'n
                            'no
                            'off
                            0}
                          (contains? x))))]
    (some-> k System/getenv .toLowerCase (string/replace #"( |'|\")" "") read-string truthy)))

(defn nedap-lib [raw-name]
  (let [default-github-username "nedap"
        github-username (or (System/getenv "NEDAP_CLJ_TEMPLATE_GITHUB_USERNAME")
                            (ask "Github user/organisation that will own the repo:"
                                 default-github-username))
        nedap? (= github-username
                  default-github-username)
        copyright-holder (or (System/getenv "NEDAP_CLJ_TEMPLATE_COPYRIGHT_HOLDER")
                             (ask "Copyright holder (example: My Company):"
                                  (when nedap?
                                    "Nedap")))
        group-id (or (System/getenv "NEDAP_CLJ_TEMPLATE_GROUP_ID")
                     (ask "Maven group id (example: com.mycompany):"
                          (when nedap?
                            "com.nedap.staffing-solutions")))
        prefix (or (System/getenv "NEDAP_CLJ_TEMPLATE_LIB_PREFIX")
                   (ask "Clojure namespace prefix:"
                        (if nedap?
                          (str "nedap." raw-name)
                          (str group-id "." raw-name))))
        gpg-key (or (System/getenv "NEDAP_CLJ_TEMPLATE_GPG_EMAIL")
                    (if nedap?
                      "releases-staffingsolutions@nedap.com"
                      (ask "Email for GPG-signing releases:"
                           nil)))
        clojurescript? (if-some [choice (env-truth "NEDAP_CLJ_TEMPLATE_USE_CLJS")]
                         choice
                         (confirmation "Also setup ClojureScript?"))
        clojars? (if-some [choice (env-truth "NEDAP_CLJ_TEMPLATE_USE_CLOJARS")]
                   choice
                   (confirmation "Deploy to Clojars?"))
        license? (if-some [choice (env-truth "NEDAP_CLJ_TEMPLATE_DO_LICENSE")]
                   choice
                   (confirmation "Publicly license?"))
        render (renderer "nedap-lib")
        escaped-prefix (string/replace prefix "." "\\.")
        main-ns (str prefix ".api")
        data {:clojars?         clojars?
              :clojurescript?   clojurescript?
              :copyright-holder copyright-holder
              :date             (date)
              :escaped-prefix   escaped-prefix
              :github-username  github-username
              :gpg-key          gpg-key
              :group-id         group-id
              :initial-release  "unreleased"
              :license?         license?
              :name             (project-name raw-name)
              :namespace        main-ns
              :nedap?           nedap?
              :nested-dirs      (name-to-path main-ns)
              :prefix           prefix
              :prefix-dir       (name-to-path prefix)
              :raw-name         raw-name
              :year             (year)}]
    (->> [["project.clj" (render "project.mustache" data)]
          ["README.md" (render "README.md" data)]
          [".gitignore" (render "gitignore" data)]
          [".github/pull_request_template.md" (render "pull_request_template.md" data)]
          [".github/PULL_REQUEST_TEMPLATE/ncrw.md" (render "ncrw_pull_request_template.md" data)]
          [".github/ISSUE_TEMPLATE/bug.md" (render "bug.md" data)]
          [".github/ISSUE_TEMPLATE/feature.md" (render "feature.md" data)]
          [".github/ISSUE_TEMPLATE/technical_improvement.md" (render "technical_improvement.md" data)]
          ["dev/dev.clj" (render "dev.clj" data)]
          [(str "src/{{nested-dirs}}.clj" (when clojurescript?
                                            "c"))
           (render "api.cljc" data)]
          [(str "test/unit/{{nested-dirs}}.clj" (when clojurescript?
                                                  "c"))
           (render "test.cljc" data)]
          (when clojurescript?
            ["test/{{prefix-dir}}/test_runner.cljs" (render "test_runner.cljs" data)])
          (when license?
            ["LICENSE" (render "LICENSE" data)])
          [".circleci/config.yml" (render "config_yml.mustache" data)]
          "resources"]
         (filter some?)
         (apply ->files data))))
