(ns leiningen.i18n.code-gen
  (:require [clojure.string :as str]
            [clojure.pprint :as pp]
            [leiningen.i18n.util :as util]
            [leiningen.core.main :as lmain]))

(defn wrap-with-swap
  "Wrap a translation map with supporting clojurescript code

  Parameters:
  * `locale` - the locale which this translation targets
  * `translation` - a clojurescript map of translations
  * `namespace` - the namespace in which to load the translations

  Returns a string of clojurescript code."
  [& {:keys [locale translation namespace]}]
  (let [trans-namespace (symbol (str namespace "." locale))
        ns-decl (pp/write (list 'ns trans-namespace
                                (list :require 'untangled.i18n.core)
                                (list :import 'goog.module.ModuleManager)) :pretty false :stream nil)
        comment ";; This file was generated by untangled's i18n leiningen plugin."
        trans-def (pp/write (list 'def 'translations translation) :stream nil)
        swap-decl (pp/write (list 'swap! 'untangled.i18n.core/*loaded-translations*
                                  (list 'fn '[x] (list 'assoc 'x locale 'translations))) :stream nil)
        goog-module-decl (pp/write '(try
                                      (-> goog.module.ModuleManager
                                          .getInstance
                                          (.setLoaded locale))
                                      (catch js/Object e)) :stream nil)]
    (str/join "\n\n" [ns-decl comment trans-def swap-decl goog-module-decl])))

(defn write-cljs-translation-file [fname translations-string]
  (lmain/info "Writing " fname)
  (spit fname translations-string))

(defn gen-default-locale-ns
  "Generate clojurescript code which initially loads translations for the project's default locale.

  Parameters:
  * `ns` - the namespace in which to load this code
  * `locale` - the default locale of the project

  Returns a string of clojurescript code."
  [ns locale]
  (let [def-lc-namespace (symbol (str ns ".default-locale"))
        translation-require (list :require (symbol (str ns "." locale)) ['untangled.i18n.core :as 'i18n])
        ns-decl (list 'ns def-lc-namespace translation-require)
        reset-decl (list 'reset! 'i18n/*current-locale* locale)
        swap-decl (list 'swap! 'i18n/*loaded-translations*
                        (symbol (str "#(assoc % :" locale " " ns "." locale "/translations)")))]
    (str/join "\n\n" [ns-decl reset-decl swap-decl])))

(defn gen-locales-ns
  "
  Generates a code string that assists in dynamically loading translations when a user changes their locale. Uses the
  i18n settings to configure the code string's namespace as well as the output directory for locale modules.

  Parameters:
  * `settings`: A leiningen project map
  * `locales`: A list of locale strings

  Returns a string of cljs code."
  [settings locales]
  (let [translation-namespace (:translation-namespace settings)
        locales-ns (-> translation-namespace (str ".locales") symbol)
        xns (fn [locale] (symbol (str translation-namespace "." locale)))
        locale-modules (:translation-modules settings)
        supports-modules? (boolean (seq locale-modules))
        is-module? (fn [l] (contains? locale-modules l))
        translations-to-include (reduce (fn [acc l]
                                          (if (is-module? l) acc (conj acc (xns l))))
                                        #{(xns (:default-locale settings))} locales)
        ns-decl (pp/write (list 'ns locales-ns
                                (concat
                                  (list :require
                                        'goog.module
                                        'goog.module.ModuleLoader
                                        '[goog.module.ModuleManager :as module-manager]
                                        '[untangled.i18n.core :as i18n])
                                  translations-to-include)
                                (list :import 'goog.module.ModuleManager)) :stream nil)
        abs-module-path (:module-server-path settings)
        manager-def (list 'defonce 'manager (list 'module-manager/getInstance))
        modules-map (reduce #(assoc %1 %2 (str abs-module-path "/" %2 ".js")) {} locales)
        modules-def (pp/write (list 'defonce 'modules (symbol (str "#js")) modules-map) :stream nil)
        mod-info-map (reduce #(assoc %1 %2 []) {} locales)
        mod-info-def (list 'defonce 'module-info (symbol (str "#js")) mod-info-map)
        loader-def (pp/write (list 'defonce (symbol "^:export")
                                   'loader '(let [loader (goog.module.ModuleLoader.)]
                                              (.setLoader manager loader)
                                              (.setAllModuleInfo manager module-info)
                                              (.setModuleUris manager modules)
                                              loader)) :stream nil)
        set-locale-def (if supports-modules?
                         (list 'defn (symbol "^:export") 'set-locale '[l]
                               '(reset! i18n/*current-locale* l)
                               '(try
                                  (.execOnLoad manager l
                                               (fn after-locale-load []
                                                 (reset! i18n/*current-locale* l)))
                                  (catch js/Object e)))
                         (list 'defn (symbol "^:export") 'set-locale '[l] '(reset! i18n/*current-locale* l)))]
    (str/join "\n\n"
              (if supports-modules?
                [ns-decl manager-def modules-def mod-info-def loader-def set-locale-def]
                [ns-decl set-locale-def]))))
