(ns leiningen.i18n.code-gen
  (:require [clojure.string :as str]
            [clojure.pprint :as pp]
            [leiningen.i18n.util :as util]))

(defn wrap-with-swap
  "Wrap a translation map with supporting clojurescript code

  Parameters:
  * `locale` - the locale which this translation targets
  * `translation` - a clojurescript map of translations
  * `namespace` - the namespace in which to load the translations

  Returns a string of clojurescript code."
  [& {:keys [locale translation namespace]}]
  (let [trans-namespace (symbol (str namespace "." locale))
        ns-decl (pp/write (list 'ns trans-namespace
                                (list :require 'untangled.i18n.core)
                                (list :import 'goog.module.ModuleManager)) :pretty false :stream nil)
        comment ";; This file was generated by untangled's i18n leiningen plugin."
        trans-def (pp/write (list 'def 'translations translation) :stream nil)
        swap-decl (pp/write (list 'swap! 'untangled.i18n.core/*loaded-translations*
                                  (list 'fn '[x] (list 'assoc 'x locale 'translations))) :stream nil)
        goog-module-decl (pp/write (list 'if (list 'exists? 'js/i18nDevMode)
                                         :noop
                                         (list '-> 'goog.module.ModuleManager '.getInstance (list '.setLoaded locale))) :stream nil)]
    (str/join "\n\n" [ns-decl comment trans-def swap-decl goog-module-decl])))

(defn write-cljs-translation-file [fname translations-string]
  (spit fname translations-string))

(defn gen-default-locale-ns
  "Generate clojurescript code which initially loads translations for the project's default locale.

  Parameters:
  * `ns` - the namespace in which to load this code
  * `locale` - the default locale of the project

  Returns a string of clojurescript code."
  [ns locale]
  (let [def-lc-namespace (symbol (str ns ".default-locale"))
        translation-require (list :require (symbol (str ns "." locale)) ['untangled.i18n.core :as 'i18n])
        ns-decl (list 'ns def-lc-namespace translation-require)
        reset-decl (list 'reset! 'i18n/*current-locale* locale)
        swap-decl (list 'swap! 'i18n/*loaded-translations*
                        (symbol (str "#(assoc % :" locale " " ns "." locale "/translations)")))]
    (str/join "\n\n" [ns-decl reset-decl swap-decl])))

(defn gen-locales-ns
  "
  Generates a code string that assists in dynamically loading translations when a user changes their locale. Uses the
  leiningen project map to configure the code string's namespace as well as the output directory for locale modules.

  Parameters:
  * `project`: A leiningen project map
  * `locales`: A list of locale strings

  Returns a string of cljs code."
  [project locales]
  (let [translation-namespace (-> project util/translation-namespace)
        locales-ns (-> translation-namespace (str ".locales") symbol)
        translations (map #(symbol ( str translation-namespace "." %)) locales)
        ns-decl (pp/write (list 'ns locales-ns
                                (concat
                                  (list :require
                                        'goog.module
                                        'goog.module.ModuleLoader
                                        '[goog.module.ModuleManager :as module-manager]
                                        '[untangled.i18n.core :as i18n]
                                        )
                                  translations
                                  )
                                (list :import 'goog.module.ModuleManager)) :stream nil)
        output-dir (:output-dir (:compiler (util/get-cljsbuild (get-in project [:cljsbuild :builds]) (util/target-build project))))
        abs-module-path (str/join (interleave (repeat "/") (drop 2 (str/split output-dir #"/"))))
        manager-def (list 'defonce 'manager (list 'module-manager/getInstance))
        modules-map (reduce #(assoc %1 %2 (str abs-module-path "/" %2 ".js")) {} locales)
        modules-def (pp/write (list 'defonce 'modules (symbol (str "#js")) modules-map) :stream nil)
        mod-info-map (reduce #(assoc %1 %2 []) {} locales)
        mod-info-def (list 'defonce 'module-info (symbol (str "#js")) mod-info-map)
        loader-def (pp/write (list 'defonce (symbol "^:export")
                                   'loader (list 'let ['loader (list 'goog.module.ModuleLoader.)]
                                                 (list '.setLoader 'manager 'loader)
                                                 (list '.setAllModuleInfo 'manager 'module-info)
                                                 (list '.setModuleUris 'manager 'modules)
                                                 'loader)) :pretty false :stream nil)
        set-locale-def (list 'defn 'set-locale ['l]
                             (list 'js/console.log (list 'str "LOADING ALTERNATE LOCALE: " 'l))
                             (list 'if (list 'exists? 'js/i18nDevMode)
                                   (list 'do (list 'js/console.log (list 'str "LOADED ALTERNATE LOCALE in dev mode: " 'l))
                                         (list 'reset! 'i18n/*current-locale* 'l)
                                         )
                                   (list '.execOnLoad 'manager 'l
                                         (list 'fn 'after-locale-load []
                                               (list 'js/console.log (list 'str "LOADED ALTERNATE LOCALE: " 'l))
                                               (list 'reset! 'i18n/*current-locale* 'l)))))]
    (str/join "\n\n" [ns-decl manager-def modules-def mod-info-def loader-def set-locale-def])))
