(ns navaratine.xlsx.utils
  (:refer-clojure :exclude [resolve])
  (:require [com.rpl.specter :as sr]))


(def xmlns
  "Excel XML namespace aliases."
  {"X"  "http://schemas.openxmlformats.org/spreadsheetml/2006/main"
   "Xr" "http://schemas.openxmlformats.org/officeDocument/2006/relationships"})


(def xmlns-encoded
  "Excel XML encoded namespace aliases."
  {"X"  "xmlns.http%3A%2F%2Fschemas.openxmlformats.org%2Fspreadsheetml%2F2006%2Fmain"
   "Xr" "xmlns.http%3A%2F%2Fschemas.openxmlformats.org%2FofficeDocument%2F2006%2Frelationships"})


(defn resolve
  "Resolve xmlns alias."
  [k]
  (keyword (xmlns-encoded (namespace k)) (name k)))


(def children
  "Navigator that returns the children of nodes."
  [:content sr/ALL])


(def attributes
  "Navigator that returns the attributes of nodes."
  :attrs)


(def text
  "Navigator that returns the text content of nodes."
  [:content sr/ALL string?])


(defn tag=
  "Navigator that filters nodes with given tag.
   Aliases to xmlns are automatically resolved."
  [k]
  (let [tag (resolve k)]
    (fn -tag= [node] (= tag (:tag node)))))


(defn attr=
  "Navigator that filters nodes with given attribute value."
  [k val]
  (fn -attr= [node] (= val (get-in node [:attrs k]))))


(def each-cell
  "Navigator that selects all cells in row."
  [children (tag= :X/c)])


(def cell-text
  "Navigator that selects text contents of a cell.
  Transformation adjust the property `t` for strings."
  (sr/nav []
          (select* [this c next-fn]
                   (let [s (sr/select [children (tag= :X/v) text] c)]
                     (next-fn s)))
          (transform* [this c next-fn]
                      (let [c2 (sr/transform [children (tag= :X/v) text] next-fn c)
                            v  (sr/select-first [children (tag= :X/v) children] c2)]
                        (if (string? v)
                          (sr/setval [:attrs :t] "s" c2)
                          (update c2 :attrs dissoc :t))))))
