(ns navaratine.xlsx
  (:require [clojure.java.io :as io]
            [clojure.data.xml :as xml]
            [clojure.string :as string]
            [clojure.pprint :refer [pprint]]
            [com.rpl.specter :as sr]
            [navaratine.xlsx.utils :as u]
            [navaratine.xlsx.shared-strings :as shared-strings]
            [navaratine.xlsx.workbook :as workbook])
  (:import (java.util.zip ZipEntry ZipInputStream ZipOutputStream)
           (java.io ByteArrayOutputStream)))


(defn- copy-zip-entry!
  "Copy entry and its content from ZipInputStream to ZipOutputStream."
  [zis zouts zen]
  (let [ozen (ZipEntry. (.getName zen))]
    (.putNextEntry zouts ozen)
    (io/copy zis zouts)
    (.closeEntry zouts)))


(defn- emit-xml-to-zip!
  "Emit XML content as entry in ZipOutputStream."
  [content filename zouts]
  (let [ozen (ZipEntry. filename)]
    (.putNextEntry zouts ozen)
    ;FIXME: Just write directly to zip stream
    ;       For some reason when I try to emit directly to writer it complains
    ;       that zip entry has been closed, I need to look into it.
    (io/copy (.getBytes (xml/emit-str content)) zouts)
    (.closeEntry zouts)))


(defn- parse-entry!
  [zis]
  ;;FIXME: If I use the reader in with-open, I get `java.io.IOException: Stream closed`
  (xml/parse (io/reader zis)))


(defn- parse-and-emit!
  "Returns a parsed XML contents of a file. 
   Identical data is emitted into ZipOutputStream."
  [zis zouts zen]
  (let [xml (parse-entry! zis)]
    (emit-xml-to-zip! xml (.getName zen) zouts)
    xml))


(defn- load-and-partially-render
  "Load and partially render the xlsx template by parsing the relevant xml files
   within archive and streaming everything else into ZipOutputStream."
  [template zouts]
  (with-open [fis (io/input-stream (io/resource template))
              zis (ZipInputStream. fis)]
    (loop [data {:shared-strings nil
                 :workbook       nil
                 :sheets         nil}
           zen  (.getNextEntry zis)]
      (if zen
        (condp re-matches (.getName zen)
          #"xl/workbook\.xml"
          (recur (assoc data :workbook (parse-and-emit! zis zouts zen))
                 (do (.closeEntry zis) (.getNextEntry zis)))
          #"xl/sharedStrings\.xml"
          (recur (assoc data :shared-strings (parse-entry! zis))
                 (do (.closeEntry zis) (.getNextEntry zis)))
          #"xl/worksheets/sheet\d+\.xml"
          (recur (assoc-in data [:sheets (.getName zen)] (parse-entry! zis))
                 (do (.closeEntry zis) (.getNextEntry zis)))
          (recur data
                 (do (copy-zip-entry! zis zouts zen)
                     (.closeEntry zis)
                     (.getNextEntry zis))))
        data))))



(defn- transform-cell
  "Replace text contents of a cell. Behaviour depends on the type of new content:
   - nil leaves the cell unchanged
   - numbers replace the cell with new value
   - strings replace the cell with new value and add `t=s` attribute
   - functions replace the cell with value returned by invoking them with cells text"
  [cell new-content]
  (cond
    (nil? new-content)
    cell
    (number? new-content)
    (sr/setval u/cell-text new-content cell)
    (string? new-content)
    (sr/setval u/cell-text new-content cell)
    (fn? new-content)
    (sr/transform u/cell-text new-content cell)))


(defn- transform-row
  "Transform row with replacements. If replacements don't contain the row index, 
   then leave it unchanged. Returns as many rows as number of replacements provided."
  [row replacements]
  (let [ridx (Integer/parseInt (get-in row [:attrs :r]))]
    (if (contains? replacements ridx)
      (for [new (get replacements ridx)]
        (sr/transform
         [(sr/subselect u/children (u/tag= :X/c)) sr/INDEXED-VALS]
         (fn [[i c]] [i (transform-cell c (get new i))])
         row))
      [row])))


(def reindex-rows
  (map-indexed
   (fn [idx row]
     (->> row
          (sr/setval [:attrs :r] (str (inc idx)))
          (sr/transform [u/children (u/tag= :X/c) :attrs :r]
                        #(string/replace % #"\d+" (str (inc idx))))))))


(defn- transform-rows
  [rows replacements decoder encoder counter]
  (sequence
   (comp (remove string?)
         decoder
         (mapcat #(transform-row % replacements))
         reindex-rows
         encoder
         counter)
   rows))


(defn- transform-sheet
  [sheet replacements decoder encoder counter]
  (sr/transform
   [(u/tag= :X/worksheet) u/children (u/tag= :X/sheetData) :content]
   #(transform-rows % replacements decoder encoder counter)
   sheet))


(defn render-stream
  [template replacements]
  (with-open [outs  (ByteArrayOutputStream.)
              zouts (ZipOutputStream. outs)]
    (let [tmpl           (load-and-partially-render template zouts)
          ss-mapping     (shared-strings/parse-shared-strings-xml (:shared-strings tmpl))
          sheet-mapping  (workbook/parse-workbook-xml (:workbook tmpl))
          decoder        (shared-strings/decoder ss-mapping)
          [encoder enc*] (shared-strings/encoder)
          [counter cnt*] (shared-strings/counter)]
      (doseq [[file name] sheet-mapping]
        (-> (get-in tmpl [:sheets file])
            (transform-sheet (replacements name)
                             decoder
                             encoder
                             counter)
            (emit-xml-to-zip! file zouts)))
      (.flush zouts) ; Ensure that encoder and counter went through all rows.
      (emit-xml-to-zip!
       (shared-strings/generate-shared-strings-xml (:shared-strings tmpl) enc* cnt*)
       "xl/sharedStrings.xml"
       zouts))
    (.finish zouts)
    outs))


(defn render-file
  [file template data]
  (with-open [s (io/input-stream (.toByteArray (render-stream template data)))
              f (io/output-stream file)]
    (io/copy s f)))
