(ns clojure-potrubi.traces.trace
  (:require  [clojure.string :as str]
             [clojure.tools.logging :as lgr]
             [clojure-carp :as carp :refer (surprise-exception)]
             [clojure-potrubi.utils.formats :as potrubi-utils-formats :refer (format-stringify-args)]
             [clojure-potrubi.utils.atoms :as potrubi-utils-atoms :refer (update-atom-one-arg maybe-update-atom-one-arg)]
             [clojure-potrubi.utils.names :as potrubi-utils-names :refer (resolve-any-class-telltale)]
             [clojure-potrubi.utils.collections :as potrubi-utils-collections :refer (to-vector is-lazyseq?)]
             [clojure-potrubi.traces.diagnostics :as carp-diag :refer (carp-diagnostics-status carp-diagnostic carp-diagnostic-format-value )]))

;; **********************
;; BEG: tracing manifests
;; **********************

(def manifest-linefeed \newline)

(def manifest-default :default)
(def manifest-format-name-default manifest-default)

(def manifest-formatters-default manifest-default)
(def manifest-enumerators-default manifest-default)
(def manifest-templates-default manifest-default)

(def manifest-formatter :formatter)
(def manifest-enumerator :enumerator)

(def manifest-format-spec-key-mnemonic :mnemonic)
(def manifest-format-spec-key-value :value)
(def manifest-format-spec-key-filter :filter)
(def manifest-format-spec-key-merger :merger)
(def manifest-format-spec-key-formatter :formatter)
(def manifest-format-spec-key-enumerator :enumerator)

(def manifest-format-spec-keys
  #{manifest-format-spec-key-mnemonic
    manifest-format-spec-key-filter
    manifest-format-spec-key-value
    manifest-format-spec-key-formatter
    manifest-format-spec-key-enumerator})

(def manifest-format-spec-fn-keys
  #{manifest-format-spec-key-formatter
    manifest-format-spec-key-enumerator})

;; **********************
;; FIN: tracing manifests
;; **********************

;; ********************
;; BEG: tracing support
;; ********************

(defn merge-format-specifications-by-mnemonics
  [specifications & mnemonics]
  (reduce (fn [s m] merge s (get specifications m)) {} (flatten mnemonics)))

(defn make-fn-merge-format-specifications
  [formatters-collection enumerators-collection]
  {:pre [(map? formatters-collection) (map? enumerators-collection)] :post [(fn? %)]}
  (fn [& {:keys [formatters enumerators]}]
    {:post [(map? %)]}
    (merge
     (merge-format-specifications-by-mnemonics formatters-collection formatters)
     (merge-format-specifications-by-mnemonics enumerators-collection  enumerators))))
(def trace-format-highlight-first-telltale-format-specification (atom "%s"))

(def trace-format-highlight-first-telltale
  (atom (fn [x] (format @trace-format-highlight-first-telltale-format-specification (str x)))))

(defn trace-format-highlight-first-arg
  [fn-highlighter & args]
  {:pre [(fn? fn-highlighter)]}
  (let [flattened-args (flatten args)
        highlighted-args (if (not-empty flattened-args)
                           (list* (fn-highlighter (first flattened-args)) (rest flattened-args)))]
    highlighted-args))

(defn trace-format-sign-args
  [trace-sign & args]
  (let [flattened-args (flatten args)
        signed-args (list (trace-format-highlight-first-arg (deref trace-format-highlight-first-telltale) (first flattened-args)) trace-sign (rest flattened-args))]
    signed-args))

(defn trace-configure
  "configure values in atoms e.g highlighter format"
  [& {:keys [first-telltale-format-specification enable-diagnostics] :as config}]
  ;;(doall (println "trace-configure" "ENTR" "CONFIG" (class config) config))
  (let []
    (maybe-update-atom-one-arg trace-format-highlight-first-telltale-format-specification first-telltale-format-specification)
    (if enable-diagnostics (update-atom-one-arg carp-diagnostics-status enable-diagnostics)))
  ;;(doall (println "trace-configure" "EXIT" "CONFIG" (class config) config))
  config)

;; ********************
;; FIN: tracing support
;; ********************

;; *****************
;; BEG: tracing misc
;; *****************

;; controls whether tracing enabled or disabled

(def initial-trace-state true)

(def trace-state (atom initial-trace-state))

(def trace-state-stack (atom [initial-trace-state]))

(defn get-trace-state [] @trace-state)

(defn set-trace-state
  [new-state]
  {:pre [(or (nil? new-state) (true? new-state) (false? new-state))]}
  (swap! trace-state (fn [& args] new-state)))

;; (defn disable-trace [& args] (swap! trace-state (fn [& args] nil)))

;; (defn enable-trace [& args] (swap! trace-state (fn [& args] true)))

(defn peek-trace-stack [& args] (peek @trace-state-stack))

(defn trace-enabled? [& args] (if (nil? (get-trace-state)) false true))

(defn trace-state-message
  [eyecatcher & telltales]
  (if (or true
          (trace-enabled?)
          )
    (let [message (format-stringify-args
                   "@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@"
                   "TRACE"
                   eyecatcher
                   "ACTIVE-STATE"
                   (get-trace-state)
                   ;;"TELLTALES"
                   telltales
                   "TRACE-STATE-STACK"
                   @trace-state-stack
                   )]
      (doall (println message)))))

(defn show-trace
  [& telltales]
  (trace-state-message "show-trace" telltales))

(defn set-trace
  [arg-state eyecatcher & telltales]
  (let [old-state (get-trace-state)
        return-state (if (nil? old-state)
                       old-state
                       (let [new-state (boolean arg-state)]
                         (if (not= new-state old-state)
                           (set-trace-state new-state))))]
    return-state))

(defn push-trace
  [new-state eyecatcher & telltales]
  ;;{:pre [(or (true? new-state) (false? new-state))]}
  (let [old-state (get-trace-state)]
    ;;(doall (println "PUSHIN>>>>>>>>>>>>>>>" "OLD-STATE" old-state "NEW-STATE" new-state "TRACE-STATE-STACK" trace-state-stack))
    (swap! trace-state-stack (fn [current-stack] (conj current-stack (boolean new-state))))
    (set-trace new-state eyecatcher telltales "PUSHED" new-state )
    ;;(doall (println "PUSHED<<<<<<<<<<<<<<<" "NEW-STATE"  new-state "TRACE-STATE-STACK" trace-state-stack))
    (get-trace-state)))

(defn pop-trace
  [eyecatcher & telltales]
  (let [old-state (get-trace-state)
        stack-state (peek-trace-stack)]
    ;;(doall (println "POPPER>>>>>>>>>>>>>>>" "OLD-STATE" old-state "STACK-STATE" stack-state "TRACE-STATE-STACK" trace-state-stack))
    (if (nil? stack-state)
      (do
        (trace-state-message eyecatcher telltales "NOT POPPED - STACK EMPTY" "STACK-STATE" stack-state "RETAINED OLD-STATE" old-state)
        old-state)
      (do
        (swap! trace-state-stack (fn [current-stack] (pop current-stack)))
        (set-trace stack-state eyecatcher telltales "POPPED" stack-state "STACK-STATE" stack-state)
        (get-trace-state)))))

(defn disable-trace
  [& args]
  ;;(doall (println "DISABLE-TRACE>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>" args))
  (if (trace-enabled?)
    (set-trace-state nil)))

(defn enable-trace
  [& args]
  ;;(doall (println "ENABLE-TRACE>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>" args))
  ;; this initial setting to enbale pop to set current state
  (swap! trace-state (fn [& args] true))
  (set-trace-state (peek-trace-stack)))

(defmacro macro-disable-trace
  [& args]
  (let [form# `(disable-trace ~@args)]
    `(do
       ~form#)))

(defmacro macro-enable-trace
  [& args]
  (let [form# `(enable-trace ~@args)]
    `(do
       ~form#)))

(defmacro macro-set-trace
  [arg-state & telltales]
  (let [form# `(set-trace ~arg-state ~@telltales)]
    `(do
       ~form#)))

(defmacro macro-push-trace
  [arg-state & telltales]
  (let [form# `(push-trace ~arg-state ~@telltales)]
    `(do
       ~form#)))

(defmacro macro-pop-trace
  [& telltales]
  (let [form# `(pop-trace ~@telltales)]
    `(do
       ~form#)))

(defn trace-mark
  "main trace diagnostic"
  [& args]
  (if @trace-state
    (let [trace-message (format-stringify-args args)]
      (doall (println trace-message))
      (lgr/debug trace-message)
      trace-message)))

;; *****************
;; FIN: tracing misc
;; *****************

;; ***********************
;; BEG: tracing validators
;; ***********************

(defn is-format-specification?
  [format-spec]
  (and (map? format-spec)
       (every? manifest-format-spec-keys (keys format-spec) )
       (every? true? (for [key-name manifest-format-spec-fn-keys :when (contains? format-spec key-name)] (fn? (get format-spec key-name))))))

(defn validate-format-specification
  [format-specification]
  {:pre [(is-format-specification? format-specification)]}
  format-specification)

;; ***********************
;; FIN: tracing validators
;; ***********************

;; **********************
;; BEG: trace enumerators
;; **********************

;; for each "loop" of the enumerator, the formatter is called on the loop value

(declare trace-formatter-class-only)

(defn- trace-value-enumerator-scalar
  [enum-value fn-formatter & args]
  {:pre [(fn? fn-formatter)]}
  (if (empty? args)
      (fn-formatter enum-value)
      (list args (fn-formatter enum-value))))

;; just adding a \newlne to each line doesn't work - spurious space at
;; beginning of 2nd and subsequent lines
(defn- trace-value-enumerator-coll
  [enum-coll fn-formatter & args]
  {:pre [(coll? enum-coll) (fn? fn-formatter)]}
  (let [telltale (format-stringify-args args (trace-formatter-class-only enum-coll))]
    (if (not-empty enum-coll)
      (str/join manifest-linefeed (map-indexed (fn [ndx kv] (format-stringify-args telltale "ndx" ndx (fn-formatter kv)))  enum-coll))
      (format-stringify-args telltale "is empty"))))

(defn- trace-value-enumerator-lazyseq
  [enum-lazyseq fn-formatter & args]
  {:pre [(is-lazyseq? enum-lazyseq) (fn? fn-formatter)]}
  (apply trace-value-enumerator-coll (doall enum-lazyseq) fn-formatter args))

;; **********************
;; FIN: trace enumerators
;; **********************

;; *********************
;; BEG: trace formatters
;; *********************

;; formatters take an item and produce a string for e.g. printing

(defn trace-formatter-full [x] (format ">%s< >%s<" (resolve-any-class-telltale x) x))
(defn trace-formatter-class-only [x] (format ">%s<" (resolve-any-class-telltale x)))
(defn trace-formatter-value-only [x] (format ">%s<" x))
(defn trace-formatter-map-kv [[k v]] (format "k >%s< v >%s< >%s<" k (resolve-any-class-telltale v) v))
(defn trace-formatter-map-kv-full [[k v]] (format "k >%s< >%s< v >%s< >%s<" (resolve-any-class-telltale k) k (resolve-any-class-telltale v) v))
(defn trace-formatter-map-kv-value-only [[k v]] (format "k >%s< v >%s<" k  v))
(defn trace-formatter-map-kv-class-only [[k v]] (format "k >%s< v >%s<" k  (resolve-any-class-telltale v)))

;; *********************
;; FIN: trace formatters
;; *********************

;; **************************************
;; BEG: tracing specification definitions
;; **************************************

(def manifest-enumerator-coll :coll)
(def manifest-enumerator-lazyseq :lazyseq)
(def manifest-enumerator-map :coll)
(def manifest-formatter-map :map-entry)

(def manifest-mnemonic-scalar :scalar)
(def manifest-mnemonic-coll :coll)
(def manifest-mnemonic-lazyseq :lazyseq)
(def manifest-mnemonic-map :map)
(def manifest-mnemonic-map-full :map-full)
(def manifest-mnemonic-map-class-only :map-class-only)
(def manifest-mnemonic-map-value-only :map-value-only)
(def manifest-mnemonic-vector :vector)
(def manifest-mnemonic-map-entry :map-entry)
(def manifest-mnemonic-map-entry-value-only :map-entry-no-class)

(def manifest-backstop-template-map manifest-mnemonic-map)
(def manifest-backstop-template-map-entry manifest-mnemonic-map-entry)
(def manifest-backstop-template-coll manifest-mnemonic-coll)

(def manifest-backstop-formatter-map manifest-mnemonic-map)
(def manifest-backstop-enumerator-coll manifest-mnemonic-coll)
(def manifest-backstop-enumerator-lazyseq manifest-mnemonic-lazyseq)

(def trace-value-enumerators
  {
   manifest-mnemonic-coll {manifest-enumerator trace-value-enumerator-coll}
   manifest-mnemonic-lazyseq {manifest-enumerator trace-value-enumerator-lazyseq}
   manifest-mnemonic-map {manifest-enumerator trace-value-enumerator-coll}
   manifest-mnemonic-vector {manifest-enumerator trace-value-enumerator-coll}
   manifest-mnemonic-map-entry {manifest-enumerator trace-value-enumerator-scalar}
   manifest-mnemonic-scalar {manifest-enumerator trace-value-enumerator-scalar}
   manifest-enumerators-default {manifest-enumerator trace-value-enumerator-scalar}
   })

(def trace-value-formatters
  {manifest-mnemonic-scalar {manifest-formatter trace-formatter-value-only}
   manifest-mnemonic-coll {manifest-formatter trace-formatter-value-only}
   manifest-mnemonic-vector {manifest-formatter trace-formatter-value-only}
   manifest-mnemonic-lazyseq {manifest-formatter trace-formatter-value-only}

   manifest-mnemonic-map {manifest-formatter trace-formatter-map-kv-value-only}

   manifest-mnemonic-map-full {manifest-formatter trace-formatter-map-kv-full}
   manifest-mnemonic-map-value-only {manifest-formatter trace-formatter-map-kv-value-only}
   manifest-mnemonic-map-class-only {manifest-formatter trace-formatter-map-kv-class-only}

   manifest-mnemonic-map-entry {manifest-formatter trace-formatter-map-kv-value-only}

   manifest-mnemonic-map-entry-value-only {manifest-formatter trace-formatter-map-kv-value-only}

   manifest-formatters-default {manifest-formatter trace-formatter-full}})

(def trace-value-formatters-default (get trace-value-formatters manifest-formatters-default))
(def trace-value-enumerators-default (get trace-value-enumerators manifest-enumerators-default))

(def merge-format-specifications (make-fn-merge-format-specifications trace-value-formatters trace-value-enumerators))

(def trace-value-templates-nom
  {

   manifest-templates-default {}

   manifest-mnemonic-scalar
   (merge-format-specifications
    :formatters manifest-mnemonic-scalar
    :enumerators manifest-mnemonic-scalar
    )

   manifest-mnemonic-map-entry
   (merge-format-specifications
    :formatters manifest-mnemonic-map-entry
    :enumerators manifest-mnemonic-map-entry
    )

   manifest-mnemonic-map-entry-value-only
   (merge-format-specifications
    :formatters manifest-mnemonic-map-entry-value-only
    :enumerators manifest-mnemonic-map-entry
    )

   manifest-mnemonic-map
   (merge-format-specifications
    :formatters manifest-mnemonic-map
    :enumerators manifest-mnemonic-map
    )

   manifest-mnemonic-map-full
   (merge-format-specifications
    :formatters manifest-mnemonic-map-full
    :enumerators manifest-mnemonic-map
    )

   manifest-mnemonic-map-value-only
   (merge-format-specifications
    :formatters manifest-mnemonic-map-value-only
    :enumerators manifest-mnemonic-map
    )

   manifest-mnemonic-map-class-only
   (merge-format-specifications
    :formatters manifest-mnemonic-map-class-only
    :enumerators manifest-mnemonic-map
    )

   manifest-mnemonic-coll
   (merge-format-specifications
    :formatters manifest-mnemonic-coll
    :enumerators manifest-mnemonic-coll
    )

   manifest-mnemonic-lazyseq
   (merge-format-specifications
    :formatters manifest-mnemonic-lazyseq
    :enumerators manifest-mnemonic-lazyseq
    )

   manifest-mnemonic-vector
   (merge-format-specifications
    :formatters manifest-mnemonic-vector
    :enumerators manifest-mnemonic-vector
    )})

(def trace-value-templates-aliases
  {
   ;;:class :kls
   ;;:map-no-class :map-no-kls
   })

(def trace-value-templates
  (reduce
   (fn [ s [k v]]
     (let [main-entry (or (get s k) (surprise-exception k "entry not found - can not be aliased"))
           aliases (to-vector v)
           alias-map (into {}  (for [alias aliases] [alias main-entry]))]
       (merge s alias-map)))
   trace-value-templates-nom
   trace-value-templates-aliases))


(def trace-value-templates-default (get trace-value-templates manifest-templates-default))

;; **************************************
;; FIN: tracing specification definitions
;; **************************************

;; ************************************
;; BEG: format specifications accessors
;; ************************************

(defn define-fn-format-specification-get-key
  [keyname telltale]
  {:pre [(keyword? keyname)] :post [(fn? %)]}
  (fn [fmt-spec]
    {:pre [(map? fmt-spec)]}
    (let [keyvalue  (get fmt-spec keyname)]
      keyvalue)))

(defn define-fn-format-specification-put-key
  [keyname]
  {:pre [(keyword? keyname)] :post [(fn? %)]}
  (fn [fmt-spec keyvalue]
    {:pre [(map? fmt-spec)] :post [(map? %)]}
    (assoc fmt-spec keyname keyvalue)))

(def format-specification-get-mnemonic (define-fn-format-specification-get-key manifest-format-spec-key-mnemonic "MNEMONIC"))
(def format-specification-get-value (define-fn-format-specification-get-key manifest-format-spec-key-value "VALUE"))
(def format-specification-get-filter (define-fn-format-specification-get-key manifest-format-spec-key-filter "FILTER"))
(def format-specification-get-formatter (define-fn-format-specification-get-key manifest-format-spec-key-formatter "FORMATTER"))
(def format-specification-get-enumerator (define-fn-format-specification-get-key manifest-format-spec-key-enumerator "ENUMERATOR"))

(def format-specification-put-mnemonic (define-fn-format-specification-put-key manifest-format-spec-key-mnemonic))
(def format-specification-put-value (define-fn-format-specification-put-key manifest-format-spec-key-value))
(def format-specification-put-filter (define-fn-format-specification-put-key manifest-format-spec-key-filter))
(def format-specification-put-formatter (define-fn-format-specification-put-key manifest-format-spec-key-formatter))
(def format-specification-put-enumerator (define-fn-format-specification-put-key manifest-format-spec-key-enumerator))

;; ************************************
;; FIN: format specifications accessors
;; ************************************

;; *********************************
;; BEG: format specification finders
;; *********************************

(defn make-fn-get-specification
  ([specifications telltale] (make-fn-get-specification specifications telltale nil))
  ([specifications telltale default]
     {:pre [(map? specifications)] :post [(fn? %)]}
     (fn [mnemonic]
       { :post [(or (nil? %) (map? %))]}
       (let [value (get specifications mnemonic default)]
         value))))

(def get-specification-enumerator (make-fn-get-specification trace-value-enumerators "ENUMERATOR"))
(def get-specification-formatter (make-fn-get-specification trace-value-formatters "FORMATTER"))
(def get-specification-template (make-fn-get-specification trace-value-templates "TEMPLATE"))

(def find-specification-enumerator (make-fn-get-specification trace-value-enumerators "ENUMERATOR" manifest-enumerators-default))
(def find-specification-formatter (make-fn-get-specification trace-value-formatters "FORMATTER" manifest-formatters-default))
(def find-specification-template (make-fn-get-specification trace-value-templates "TEMPLATE" manifest-templates-default))

;; *********************************
;; BEG: format specification finders
;; *********************************

;; **********************************
;; BEG: specification resolver makers
;; **********************************

(defn make-fn-find-format-specification-wrapper
  "makes a wrapper function that takes a format-spec map,
   gets a value from it using the fn-find-arg fn
   and calls the fn-find-spec fn with the found value to get the result spec"
  [fn-find-arg fn-find-spec telltale]
  {:pre [(fn? fn-find-arg) (fn? fn-find-spec)] :post [(fn? %)]}
  (fn  [format-spec]
    {:pre [(map? format-spec)] :post [(or (nil? %) (map? %))]}
    (let [find-arg (fn-find-arg format-spec)
          find-spec (fn-find-spec find-arg)]
      find-spec)))

(defn make-fn-resolve-format-specification
  "makes a fn to call a find specification fn
   with the passed format-spec as argument
   IF predicate function returns NIL.
   the predicate can be used to e.g. check if a formatter has already been found.
   merges the found spec with the passed spec and returns the merged map"
  [fn-find-spec fn-find-pred telltale]
  {:pre [(fn? fn-find-spec) (fn? fn-find-pred)] :post [(fn? %)]}
  (fn  [format-spec]
    {:pre [(map? format-spec)] :post [(map? %)]}
    (let [find-spec (if-not (fn-find-pred format-spec) (fn-find-spec format-spec))
          updated-format-spec
          (if-not find-spec
            format-spec
            (if (contains? find-spec manifest-format-spec-key-merger)
              ((get find-spec manifest-format-spec-key-merger) format-spec find-spec)
              (merge format-spec find-spec)) )]
      updated-format-spec)))

(defn predicate-always-find-format-specification [& args]
  nil)

;; **********************************
;; FIN: specification resolver makers
;; **********************************

;; ***********************************
;; BEG: find specification by mnemonic
;; ***********************************

(defn make-fn-find-format-specification-by-mnemonic
  [fn-find-spec telltale]
  {:pre [(fn? fn-find-spec)] :post [(fn? %)]}
  (fn [mnemonic]
    {:pre [(or (nil? mnemonic) (keyword? mnemonic))] :post [(or (nil? %) (map? %))]}
    (let [find-spec  (fn-find-spec mnemonic)]
      find-spec)))

(def find-formatter-by-mnemonic (make-fn-find-format-specification-by-mnemonic get-specification-formatter "find-formatter-by-mnemoni"))
(def find-enumerator-by-mnemonic (make-fn-find-format-specification-by-mnemonic get-specification-enumerator "find-enumerator-by-mnemonic"))

(def find-template-by-mnemonic-nom (make-fn-find-format-specification-by-mnemonic get-specification-template "find-template-by-mnemonic-nom"))

;; need to add a default clause
(defn find-template-by-mnemonic
  [mnemonic]
  {:pre [(or (nil? mnemonic) (keyword? mnemonic))] :post [(or (nil? %) (map? %))]}
  (let [find-spec-nom (if mnemonic (find-template-by-mnemonic-nom mnemonic))
        find-spec (or find-spec-nom trace-value-templates-default)]
    find-spec))

;; ***********************************
;; BEG: find specification by mnemonic
;; ***********************************

;; *************************************
;; BEG: find backstop specification by value
;; *************************************

(defn find-backstop-formatter-by-value
  [value]
  {:post [(map? %)]}
  (cond
   (map? value) (find-specification-formatter manifest-backstop-formatter-map)
   :else trace-value-formatters-default))

(defn find-backstop-enumerator-by-value
  [value]
  {:post [(map? %)]}
  (cond
   (coll? value) (find-specification-enumerator manifest-backstop-enumerator-coll)
   (is-lazyseq? value) (find-specification-enumerator manifest-backstop-enumerator-lazyseq)
   :else trace-value-enumerators-default))

(defn find-backstop-template-by-value
  [value]
  {:post [(map? %)]}
  (cond
   (map? value) (find-specification-template manifest-backstop-template-map)
   (instance? clojure.lang.MapEntry value) (find-specification-template manifest-backstop-template-map-entry)
   (coll? value) (find-specification-template manifest-backstop-template-coll)
   :else trace-value-templates-default))

;; *************************************
;; FIN: find backstop specification by value
;; *************************************

(def find-format-specification-backstop-formatter (make-fn-find-format-specification-wrapper
                                                   format-specification-get-value
                                                   find-backstop-formatter-by-value
                                                   "BACKSTOP-FORMATTER"))

(def find-format-specification-backstop-enumerator (make-fn-find-format-specification-wrapper
                                                    format-specification-get-value
                                                    find-backstop-enumerator-by-value
                                                    "BACKSTOP-ENUMERATOR"))

(def find-format-specification-backstop-template (make-fn-find-format-specification-wrapper
                                                  format-specification-get-value
                                                  find-backstop-template-by-value
                                                  "BACKSTOP-TEMPLATE"))

(def find-format-specification-mnemonic-formatter (make-fn-find-format-specification-wrapper
                                                   format-specification-get-mnemonic
                                                   find-formatter-by-mnemonic
                                                   "MNEMONIC-FORMATTER"))

(def find-format-specification-mnemonic-enumerator (make-fn-find-format-specification-wrapper
                                                    format-specification-get-mnemonic
                                                    find-enumerator-by-mnemonic
                                                    "MNEMONIC-ENUMERATOR"))

(def find-format-specification-mnemonic-template (make-fn-find-format-specification-wrapper
                                                  format-specification-get-mnemonic
                                                  find-template-by-mnemonic
                                                  "MNEMONIC-TEMPLATE"))

;; **************************
;; BEG: specification finders
;; **************************

;; ****************************
;; BEG: specification resolvers
;; ****************************

(def resolve-format-specification-backstop-formatter (make-fn-resolve-format-specification
                                                      find-format-specification-backstop-formatter
                                                      ;;get-specification-formatter
                                                      format-specification-get-formatter
                                                      "BACKSTOP-FORMATTER"))

(def resolve-format-specification-backstop-enumerator (make-fn-resolve-format-specification
                                                       find-format-specification-backstop-enumerator
                                                       ;;get-specification-enumerator
                                                       format-specification-get-enumerator
                                                       "BACKSTOP-ENUMERATOR"))

(def resolve-format-specification-backstop-template (make-fn-resolve-format-specification
                                                     find-format-specification-backstop-template
                                                     get-specification-template
                                                     ;;format-specification-get-template
                                                     "BACKSTOP-TEMPLATE"))

(def resolve-format-specification-formatter (make-fn-resolve-format-specification
                                             find-format-specification-mnemonic-formatter
                                             format-specification-get-formatter
                                             "FORMATTER"))

(def resolve-format-specification-enumerator (make-fn-resolve-format-specification
                                              find-format-specification-mnemonic-enumerator
                                              format-specification-get-enumerator
                                              "ENUMERATOR"))

(def resolve-format-specification-template (make-fn-resolve-format-specification
                                            find-format-specification-mnemonic-template
                                            predicate-always-find-format-specification
                                            "TEMPLATE"))

;; ****************************
;; FIN: specification resolvers
;; ****************************

;; ************
;; BEG: filters
;; ************

(def filter-directives-dictionary
  {:select-keys 1
   :filter-keys 1
   :telltale 1})

(defn apply-filters-resolve-directives
  [filter-directives]
  {:pre [(vector? filter-directives)] :post [(map? %)]}
  (let [resolved-directives
        (apply array-map
               (loop [kv-pairs [] directives filter-directives]
                 (if (and (coll? directives) (not-empty directives))
                   (let [directive (first directives)

                         ;; how many args for this directive
                         directive-args-count (get filter-directives-dictionary directive 0)

                         directive-args (cond
                                         (= 0 directive-args-count) nil
                                         (= 1 directive-args-count) (first (rest directives))
                                         :else (take directive-args-count (rest directives)))]

                     (recur (concat kv-pairs [directive directive-args]) (drop (+ 1 directive-args-count) directives)))

                   kv-pairs)))]

    resolved-directives
    ))

(defn apply-filters
  [format-spec]
  {:pre [(map? format-spec)] :post [(map? %)]}
  (let [filter-spec-nom (format-specification-get-filter format-spec)
        updated-format-spec (if-not filter-spec-nom
                              format-spec
                              (let [filter-directives (apply-filters-resolve-directives (to-vector filter-spec-nom))
                                    filter-value (format-specification-get-value format-spec)

                                    updated-filter-value
                                    (reduce (fn [s [fn-name fn-args]]
                                              (if fn-args
                                                (apply fn-name fn-args)
                                                (fn-name s)))
                                            filter-value
                                            filter-directives)]
                                (format-specification-put-value format-spec updated-filter-value)
                                ;;nil ; force error for now
                                ))]
    updated-format-spec))

;; ************
;; FIN: filters
;; ************

(defn trace-value-resolve-specification
  [format-spec]
  {:pre [(map? format-spec)] :post [(map? %)]}
  (-> format-spec
      validate-format-specification
      apply-filters
      resolve-format-specification-template
      ;;resolve-format-specification-backstop-template
      resolve-format-specification-formatter
      resolve-format-specification-backstop-formatter
      resolve-format-specification-enumerator
      resolve-format-specification-backstop-enumerator))

(defn normalise-format-specification
  [format-spec-nom]
  {:post [(or (nil? %) (map? %))]}
  (let [format-spec (cond
                     (map? format-spec-nom) format-spec-nom
                     (keyword? format-spec-nom) {manifest-format-spec-key-mnemonic format-spec-nom}
                     (string? format-spec-nom) nil
                     :else (surprise-exception format-spec-nom "format-spec is what?"))]
    (if format-spec (validate-format-specification format-spec))))

;; ****************
;; FIN: format spec
;; ****************

;; *****************
;; BEG: trace values
;; *****************

(defn trace-value-format
  "main fn to format a value
  the format-name args is interpreted as the format to use
  but if not found, its assumed part of args i.e. telltales
  the value is used to find the enumerator e.g. for a collection or scalar (default)
  returns a list of elements to be e.g. printed"
  ([value] (trace-value-format value manifest-format-name-default))
  ([value & args]
     (let [flatten-args (flatten args)

           format-spec-maybe (normalise-format-specification (first flatten-args))

           format-spec-norm (or format-spec-maybe {})

           ;; if spec found, remove from args to prevent it printing
           normalised-args (if format-spec-maybe
                             (apply trace-format-highlight-first-arg @trace-format-highlight-first-telltale (rest flatten-args))
                             (apply trace-format-highlight-first-arg @trace-format-highlight-first-telltale flatten-args))

           format-spec (trace-value-resolve-specification (format-specification-put-value format-spec-norm value))

           ;;normalised-args ["<<<<<<<<<<<<<<<<<<<<<<<<>>>>>>>>>>>>>>>>>>>>>>" "norm" "args"]

           ;; ;; now the enumerator - use the one in the format spec else find one

           fn-formatter (format-specification-get-formatter format-spec)
           fn-enumerator (format-specification-get-enumerator format-spec)
           normalised-value (format-specification-get-value format-spec)]

       (apply fn-enumerator normalised-value  fn-formatter normalised-args))
     ))

(defn trace-value
  ([value] (trace-value value manifest-format-name-default))
  ([value & args]
     ;;(show-trace "trace-value"  "TRACE-STATE" trace-state (class  @trace-state)  @trace-state "VALUE" value)
     (if @trace-state (trace-mark (trace-value-format value args)))
     ;; alays return the passed value for fluent interfaces
     value))

;; *****************
;; FIN: trace values
;; *****************

;; **************************
;; BEG: packaged trace points
;; **************************

(def trace-entr-sign ">>>>ENTR")
(def trace-exit-sign "EXIT<<<<")
(def trace-call-sign "<<CALL>>")
(def trace-body-sign "  BODY  ")

(def trace-entr-telltale "ENTR")
(def trace-exit-telltale "EXIT")
(def trace-call-telltale "CALL")
(def trace-body-telltale "BODY")

(defn make-fn-trace-signed
  [trace-sign trace-telltale]
  (fn [& args]
    (if @trace-state
      (do
        (trace-mark (trace-format-sign-args trace-sign args))))))

(def trace-entr (make-fn-trace-signed trace-entr-sign trace-entr-telltale))
(def trace-exit (make-fn-trace-signed trace-exit-sign trace-exit-telltale))
(def trace-call (make-fn-trace-signed trace-call-sign trace-call-telltale))
(def trace-body (make-fn-trace-signed trace-body-sign trace-body-telltale ))

(defn make-fn-trace-value-signed
  [trace-sign trace-telltale]
  (fn  [value mnemonic & args]
    ;;(show-trace "fn-trace-value-signed" trace-telltale "TRACE-STATE" trace-state (class @trace-state) @trace-state "VALUE" value "MNEMONIC" (class mnemonic) mnemonic)
    (if @trace-state
      (cond
       ;; commonest case - same as else
       (string? mnemonic) (trace-value value (apply trace-format-sign-args trace-sign mnemonic args))
       (map? mnemonic) (apply trace-value value mnemonic (trace-format-sign-args trace-sign args))
       (and (keyword? mnemonic) (get-specification-template mnemonic)) (apply trace-value value mnemonic (trace-format-sign-args trace-sign args))
       :else (trace-value value (apply trace-format-sign-args trace-sign mnemonic args))))))

(def trace-value-entr (make-fn-trace-value-signed trace-entr-sign trace-entr-telltale))
(def trace-value-exit (make-fn-trace-value-signed trace-exit-sign trace-exit-telltale))
(def trace-value-call (make-fn-trace-value-signed trace-call-sign trace-call-telltale))
(def trace-value-body (make-fn-trace-value-signed trace-body-sign trace-body-telltale))

;; **************************
;; FIN: packaged trace points
;; **************************

