(ns clojure-potrubi.tests.harnesses
  (:use [clojure.test :only [deftest is function?]])
  (:require [clojure-potrubi.traces.diagnostics :as carp-diag :refer (macro-set-diagnostics)]))

;; Low Level Testing Harnesses

;; self-contained except for diagnosticvs

;; *************************************
;; BEG: misc exception support functions
;; *************************************

(defn to-collection
  [any]
  (cond
   (coll? any) any
   (nil? any) (list)
   :else (list any)))

(defn to-vector
  [any]
  (cond
   (vector? any) any
   (nil? any) []
   :else [any]))

(defn zip-keys-and-values
  [arg-keys arg-values]
  {:pre [(coll? arg-keys) (coll? arg-values) (>= (count arg-keys) (count arg-values)  )] :post [(map? %)]}
  (zipmap arg-keys arg-values))

(defn format-stringify-args
  [& args]
  (let [normalised-args (flatten args)]
    #_(doall (for [arg normalised-args] (doall (println (format "format-stringify-args ARG ^%s^ ^%s^" (class arg) arg)))))
    (apply str (interpose " " normalised-args))))

(defn format-exception-value [value] (str "Value " (format ">%s< >%s<" (class value) value)))
(defn format-assertion-value [value] (str "Value " (format ">%s< >%s<" (class value) value)))

(defn report-any
  [& args]
  (doall (println (apply format-stringify-args args))))

;; *************************************
;; FIN: misc exception support functions
;; *************************************

;; ***************
;; BEG: exceptions
;; ***************

(defn raise-exception
  "throw an exception"
  [& args]
  (let [exception-message (format-stringify-args args)]
    (doall (println "RAISE EXCEPTION" exception-message))
    (throw (Exception. exception-message))))

(defn surprise-exception
  [value & args]
  (raise-exception "Surprise Exception" (format-exception-value value) args))

;; ***************
;; FIN: exceptions
;; ***************

(defn run-function
  [fn-name & fn-args]
  {:pre [(fn? fn-name)]}
  (let [fn-result (apply fn-name fn-args)]
    (report-any "run-function" "RESULT" fn-result "FN-ARGS" fn-args)
    fn-result))

;; *******************
;; BEG: test harnesses
;; *******************

(def test-harness1-keys
  #{:fn :test-status :args :test-result})

(defn test-harness1
  [& {:keys [fn args test-status test-result] :as opt-args}]
  (report-any  "test-harness1" "ENTR" "OPT-ARGS" opt-args)
  (let [actual-result nil

        _ (assert (every? test-harness1-keys (keys opt-args)))

        test-status (get opt-args :test-status)
        test-result (get opt-args :test-result)
        fn-test (get opt-args :fn)
        fn-args (to-vector (get opt-args :args))

        function-result  (condp = test-status
                                    :ok (apply run-function fn-test fn-args)
                                    :not (apply run-function fn-test fn-args))

        _ (doall (println "test-harness1" "TEST-RESULT" (class test-result) test-result))
        _ (doall (println "test-harness1" "FUNC-RESULT" (class function-result) function-result))

        harness-result (condp = test-status
                        :ok (and
                              (is (= test-result function-result))
                              function-result)
                        :not (and
                               (is (not= test-result function-result))
                               function-result)
                        :error (is (thrown? AssertionError (apply run-function fn-test fn-args)))
                        :exception (is (thrown? Exception (apply run-function fn-test fn-args)))
                        (surprise-exception test-status "test-status is what?"))]
    (report-any  "test-harness1" "EXIT" "HARNESS-RESULT" harness-result "TEST-RESULT" (class test-result) test-result "FUNCTION-RESULT" function-result "OPT-ARGS" opt-args)
    harness-result))

;; *******************
;; FIN: test harnesses
;; *******************

;; **************************************
;; BEG: will-work and will-fail harnesses
;; **************************************

;; Helper for accessor examples expected to work.  Returns the expected result, else fails

(defn will-work
  [fn-constrained & fn-args]
  (let [actual-result (apply fn-constrained fn-args)]
    (println "will-work" "worked as expected" "actual-result" actual-result "fn-constrained" fn-constrained "fn-args" fn-args)
    actual-result))
;; A nil return from the function is ok

(defn will-fail
  [fn-constrained & fn-args]
  (try
    (do
      (let [return-value (apply fn-constrained fn-args)]
        (if return-value (assert (println "will-fail" "DID NOT FAIL" "did not cause AssertionError" "fn-constrained" fn-constrained "fn-args" fn-args "RETURN-VALUE" (class return-value) return-value)))))
    (catch AssertionError e
      (println "will-fail" "failed as expected" "fn-constrained" fn-constrained "fn-args" fn-args))))

;; **************************************
;; FIN: will-work and will-fail harnesses
;; **************************************
