(ns naipmoro.qrng.core
  (:require [clj-http.client :as http]))

(def anu-domain "qrng.anu.edu.au/API/jsonI.php")
(def ^:const maxlen 1024) ; determined by the ANU server
(def types {:int8 "uint8", :int16 "uint16", :hex16 "hex16"})

(defn quot-rem [num denom]
  [(quot num denom) (rem num denom)])

(defn qrand
  "Returns a vector of random numbers generated by the
   ANU Random Numbers Server.

   The ':length' keyword determines the number of random numbers to be
   returned. The default is 1.

   You have 3 choices for keyword ':type':
   :int8  (returns integers between 0-255; the default)
   :int16 (returns integers between 0-65535)
   :hex16 (returns hexadecimal strings between 00-ff)

   The ':blocks' keyword is relevant only for type :hex16. It sets the
   hexadecimal block size and must be a number between 1-1024. The default
   is 1.

   If keyword ':https' is true, the connection will be established using
   the https protocol. The default is false (http is used).

   If the requested vector is larger than the server's maximum (1024),
   repeated requests need to be made. A connection pool is used in the
   hope of increasing throughput. Unfortunately, actual experience
   shows no improvement. Perhaps the ANU server disallows persistent
   connections -- or something else is going on."
  
  [& {:keys [type length blocks https]
      :or {type   :int8     ; 
           length 1         ; the defaults
           blocks 1         ; 
           https  false}}]  ;
  {:pre [(pos? length)                        ; length > 0
         (and (> blocks 0) (< blocks 1025))]} ; 0 < blocks <= 1024
  (let [protocol (if https "https" "http")
        url (str protocol "://" anu-domain)
        base-query {"type" (type types) "size" blocks}
        base-get {:as :json}]
    (if (> length maxlen)
      (let [query-opts {:query-params (assoc base-query "length" maxlen)}
            opts (conj base-get query-opts)]
        (http/with-connection-pool {:insecure? true}
          (let [[q r] (quot-rem length maxlen), v (atom [])]
            (dotimes [_ q] (swap! v concat (:data (:body (http/get url opts)))))
            (when (> r 0)
              (swap! v concat (take r (:data (:body (http/get url opts)))))) 
            (vec (deref v)))))
       ; else if length <= maxlen
      (let [query-opts {:query-params (assoc base-query "length" length)}
            get-opts (if-not https base-get (assoc base-get :insecure? true))
            opts (conj get-opts query-opts)]
        (:data (:body (http/get url opts)))))))

