(ns datomic-to-schema-core.core
  (:require [clojure.edn :as edn]
            [clojure.string :as string]
            [schema.core :as s]
            ))

(def datomic-value-type->schema-value-type
  {:db.type/string s/Str})

(defn schema-core-map-key-heirarchy [datomic-ident]
  (let [schema-name-and-key (last (string/split (str datomic-ident) #"\."))]
    (vec (map keyword (string/split schema-name-and-key #"/")))))

(defn schema-core-map-value [value-type cardinality]
  (let [schema-value-type (value-type datomic-value-type->schema-value-type
                                      "error : no value type")]
    (case cardinality
      :db.cardinality/one schema-value-type
      :db.cardinality/many [schema-value-type]
      "error")))

(defn datomic-map->schema-core-map [existing-map {ident :db/ident
                                                  value-type :db/valueType
                                                  cardinality :db/cardinality}]
  (assoc-in existing-map
            (schema-core-map-key-heirarchy ident)
            (schema-core-map-value value-type cardinality)))

(defn schema-core-map->schema-core-schema-as-string [[schema-name-as-key schema]]
  (let [schema-name (name schema-name-as-key)]
    (str "(s/defschema "
         (string/capitalize (subs schema-name 0 1))
         (subs schema-name 1)
         " "
         (str schema)
         ")")))

(defn define-schema [schema-core-map]
  (-> schema-core-map
      schema-core-map->schema-core-schema-as-string
      read-string
      eval))

(defn define-schemas [schema-core-map-of-maps]
  (map define-schema schema-core-map-of-maps))

(defn convert-string [input]
  (let [datomic-vector-of-maps (edn/read-string input)]
    (reduce datomic-map->schema-core-map {} datomic-vector-of-maps)))

(defn convert-and-prettify-schema-core-map [schema-core-map]
  (let [quoted-str (-> schema-core-map
                       schema-core-map->schema-core-schema-as-string
                       clojure.pprint/pprint
                       with-out-str)]
    (subs quoted-str 1 (- (count quoted-str) 2))))

(defn convert-file [input-file output-file]
  (let [schema-core-map-of-maps (convert-string (slurp input-file))]
    (spit output-file  (string/join "\n"  (map convert-and-prettify-schema-core-map
                                               schema-core-map-of-maps)))
    (define-schemas schema-core-map-of-maps)))
