
(ns respo-value.component.value
  (:require [hsl.core :refer [hsl]]
            [respo-value.style.widget :as widget]))

(defn style-number []
  (merge widget/literal {:background-color (hsl 240 80 60)}))

(defn style-nil []
  (merge widget/literal {:background-color (hsl 240 80 0)}))

(defn style-string []
  (merge widget/literal {:background-color (hsl 140 80 40)}))

(defn style-keyword []
  (merge widget/literal {:background-color (hsl 200 60 80)}))

(defn style-bool []
  (merge widget/literal {:background (hsl 140 80 50)}))

(defn style-vector []
  (merge
    widget/literal
    {:background-color (hsl 0 20 80), :cursor "pointer"}))

(def style-char {:color (hsl 0 80 30), :pointer-events "none"})

(defn style-list []
  (merge
    widget/literal
    {:background-color (hsl 120 80 70), :cursor "pointer"}))

(defn style-map []
  (merge
    widget/literal
    {:background-color (hsl 230 50 80), :cursor "pointer"}))

(defn style-set []
  (merge
    widget/literal
    {:background-color (hsl 230 50 70), :cursor "pointer"}))

(defn style-function []
  (merge widget/literal {:background-color (hsl 30 50 80)}))

(defn style-unknown [] {})

(declare render-value)

(declare render-children)

(def nil-component
 {:name :number,
  :get-state (fn [] {}),
  :render
  (fn [] (fn [state] [:span {:inner-text "nil", :style (style-nil)}])),
  :update-state merge})

(def function-component
 {:name :function,
  :get-state (fn [] {}),
  :render
  (fn []
    (fn [state] [:span {:inner-text "fn", :style (style-function)}])),
  :update-state merge})

(def number-component
 {:name :number,
  :get-state (fn [x] {}),
  :render
  (fn [x]
    (fn [state] [:span {:inner-text (str x), :style (style-number)}])),
  :update-state merge})

(def string-component
 {:name :string,
  :get-state (fn [x] {}),
  :render
  (fn [x]
    (fn [state] [:span
                 {:inner-text (pr-str x), :style (style-string)}])),
  :update-state merge})

(def keyword-component
 {:name :keyword,
  :get-state (fn [x] {}),
  :render
  (fn [x]
    (fn [state] [:span
                 {:inner-text (str x), :style (style-keyword)}])),
  :update-state merge})

(def bool-component
 {:name :bool,
  :get-state (fn [x] {}),
  :render
  (fn [x]
    (fn [state] [:span {:inner-text (str x), :style (style-bool)}])),
  :update-state merge})

(defn toggle-folding [simple-event dispatch mutate] (mutate))

(def vector-component
 {:name :vector,
  :get-state (fn [x level] (> level 1)),
  :render
  (fn [x level]
    (fn [folded?]
      (if folded?
        [:div
         {:on-click toggle-folding, :style (style-vector)}
         [:span
          {:inner-text (str "Vector:" (count x)),
           :style widget/only-text}]]
        [:div
         {:on-click toggle-folding, :style (style-vector)}
         [:span {:inner-text "[", :style style-char}]
         [:span {:inner-text "]", :style style-char}]
         (render-children x level)]))),
  :update-state not})

(def set-component
 {:name :set,
  :get-state (fn [x level] (> level 1)),
  :render
  (fn [x level]
    (fn [folded?]
      (if folded?
        [:div
         {:on-click toggle-folding, :style (style-set)}
         [:span
          {:inner-text (str "Set:" (count x)),
           :style widget/only-text}]]
        [:div
         {:on-click toggle-folding, :style (style-set)}
         [:span {:inner-text "#{", :style style-char}]
         [:span {:inner-text "}", :style style-char}]
         (render-children x level)]))),
  :update-state not})

(def list-component
 {:name :list,
  :get-state (fn [x level] (> level 1)),
  :render
  (fn [x level]
    (fn [folded?]
      (if (not folded?)
        [:div
         {:on-click toggle-folding, :style (style-list)}
         [:span {:inner-text "'(", :style style-char}]
         [:span {:inner-text ")", :style style-char}]
         (render-children x level)]
        [:div
         {:on-click toggle-folding, :style (style-list)}
         [:span
          {:inner-text (str "List:" (count x)),
           :style widget/only-text}]]))),
  :update-state not})

(def map-component
 {:name :map,
  :get-state (fn [x level] (> level 1)),
  :render
  (fn [x level]
    (fn [folded?]
      (if folded?
        [:div
         {:on-click toggle-folding, :style (style-map)}
         [:span
          {:inner-text (str "Map:" (count x)),
           :style widget/only-text}]]
        [:div
         {:on-click toggle-folding, :style (style-map)}
         [:span {:inner-text "{", :style style-char}]
         [:span {:inner-text "}", :style style-char}]
         (render-children
           (->> x (map identity) (apply concat))
           level)]))),
  :update-state not})

(def style-children
 {:vertical-align "top", :padding "4px 4px", :display "inline-block"})

(defn render-children [xs level]
  [:div
   {:style style-children}
   (->>
     xs
     (map-indexed
       (fn [index child] [index (render-value child (inc level))]))
     (into (sorted-map)))])

(defn render-value
  ([x] (render-value x 0))
  ([x level]
    (cond
      (nil? x) [nil-component]
      (number? x) [number-component x]
      (string? x) [string-component x]
      (keyword? x) [keyword-component x]
      (fn? x) [function-component x]
      (or (= x true) (= x false)) [bool-component x]
      (vector? x) [vector-component x level]
      (set? x) [set-component x level]
      (list? x) [list-component x level]
      (map? x) [map-component x level]
      :else [:div {:inner-text "unknown", :style style-unknown}])))
