(ns weir.core
  "The core API."
  (:require-macros [reagent.ratom :as rm])
  (:require [datascript.core :as d]
            [franz.core :as f]
            [reagent.core :as r]
            [taoensso.sente :as s]))

(defmulti event-key
  (fn [event] event))

(defmethod event-key :default
  [_]
  nil)

(defmulti event-handler
  (fn [event data context]
    event))

(defn- watch-db
  [app-db]
  (fn [_ _ _ new-db] (reset! app-db new-db)))

(defn- watch-franz
  [log-fn context event-handler]
  (f/sparse-handler
   (fn [offset [key [event data]]]
     (log-fn "Got event message:" event)
     (event-handler event data context))))

(defn- sente-router
  [log-fn topic]
  (fn [msg]
    (let [{:keys [id ?data]} msg]
      (if (= :chsk/recv id)
        (let [[event data] ?data]
          (f/send! topic (event-key event) [event data]))
        (log-fn "Got non-event sente message:" id)))))

(defn ^:export initialize!
  "This function sets everything in motion. It will set everything up
  and connect sente to your server. It takes no required arguments,
  though options can be specified using keyword arguments.

  It returns a map, with several entries, but the important two are
  `:app-db` and `:emit-fn`. The values of those can be passed to your
  view components, as soon as you mount them. For example:

  ```
  (defn ^:export mount []
    (let [weir (weir/initialize!)]
      (r/render-component [my-component (:app-db weir) (:emit-fn weir)]
                          (.getElementById js/document \"my-id\"))))
  ```

  The `initialize!` function can take the following keyword arguments:

  * `:sente-path` - The path the sente client should connect to on the
  server. Default is `\"/chsk\"`. Set this to `nil` to skip starting
  sente at all.

  * `:sente-opts` - The options for the sente client. Default is
  `{:type :auto}`.

  * `:schema` - The DataScript schema. Default is `{}`.

  * `:init-tx` - An initial DataScript transaction. Default is
  `nil` (no transaction).

  * `:topic-opts` - The options passed when creating the franz topic.
  This is a more advanced feature; the defaults should work quite
  well.

  * `:log-fn` - A variable arity function, taking objects forming a
  debug message. Expects the objects to be interposed with a space.
  Default is `println`.

  * `:event-handler` - (fn [event data context] ..). by default the
  weir.core/event-handler multimethod is used
"
  [& {:keys [schema init-tx topic-opts sente-path sente-opts log-fn event-handler]
      :or   {sente-path "/chsk"
             sente-opts {:type :auto}
             log-fn     println
             event-handler weir.core/event-handler}}]
  (let [conn    (d/create-conn schema)
        app-db  (r/atom (d/db conn))
        topic   (f/topic nil topic-opts)
        emit-fn (fn emit-event
                  ([event]
                   (emit-event event nil))
                  ([event data]
                   (f/send! topic (event-key event) [event data])))
        sente   (when sente-path
                  (s/make-channel-socket! sente-path sente-opts))
        context {:conn conn :emit-fn emit-fn :sente! sente}]
    (add-watch conn :watch-db (watch-db app-db))
    (when init-tx
      (d/transact! conn init-tx))
    (defmethod weir.core/event-handler :default
      [event _ _]
      (log-fn "Unhandled event:" event))
    (f/subscribe! topic :watch-franz (watch-franz log-fn context event-handler))
    (let [stop-router-fn (when sente
                           (s/start-client-chsk-router! (:ch-recv sente)
                                                        (sente-router log-fn topic)))
          stop-sente-fn  (when sente
                           #(s/chsk-disconnect! (:chsk sente)))
          stop-fn        #(when sente
                            (stop-sente-fn) (stop-router-fn))]
      {:app-db  app-db
       :emit-fn emit-fn
       :sente!  sente
       :topic!  topic
       :conn!   conn
       :stop-fn stop-fn})))
