(ns org.hellodata.api.request
  "Functions that build (but not send) HTTP requests."
  (:require [cheshire.core :as json]
            [clojure.string :as string]
            [org.hellodata.crypto :as crypto]))

(defn- request-method-str
  [{:keys [request-method method] :as request}]
  {:pre [(or request-method method)]}
  (string/upper-case (name (or request-method method))))

(defn- request-path
  [{:keys [url uri] :as request}]
  {:pre [(or url uri)]}
  (or uri (.getPath (java.net.URL. url))))

(defn sign
  "Sign a ring or http-kit style request using `rsa-key`.
  See `org.hellodata.crypto/private-key`"
  [{:keys [url body] :as request} rsa-key]
  (assoc-in request [:headers "X-Auth-Sig"]
            (crypto/sign-string rsa-key
                                (str (request-method-str request)
                                     (request-path request)
                                     (if body body "")))))

(defn authenticate
  "Set X-Auth-User header and sign request with `rsa-key`"
  [request identifier rsa-key]
  (-> request
      (assoc-in [:headers "X-Auth-User"] identifier)
      (sign rsa-key)))

(defn json-body
  "Assoc `data` as json encoded request body and setup headers"
  [request data]
  (-> request
      (assoc-in [:headers "Content-Type"] "application/json")
      (assoc :body (json/generate-string data))))

(defn accept
  "Add accept-term to accept header. Creates accept header if
  there isn't one already."
  [request accept-term]
  (update-in request [:headers "Accept"]
             #(if % (str % ", " accept-term) accept-term)))

(defn accept-json
  "Setup headers and config for request to return an automatically
  decoded json response"
  [request]
  (-> request
      (assoc :as :json)
      (accept "application/json")))

(defn accept-api-v1
  "Specify HelloData.org SPA API version 1"
  ([request]
   (accept request "application/vnd.hd.spa-v1"))
  ([]
   (accept-api-v1 {})))

(defn request
  [method url]
  {:request-method method
   :url url
   :throw-exceptions false})

(defn format-instant
  "Format an instant as RFC3339."
  [instant]
  (-> (with-out-str (print instant))
      (string/replace #"#inst " "")
      (string/replace #"\"" "")))

