(ns org.hellodata.api.spa
  (:require [org.hellodata.api.request :as request]
            [org.hellodata.api.oauth :as oauth]))

(defn valid-config?
  [{:keys [endpoint-url callback-url spa-identifier shared-secret rsa-key]}]
  (and rsa-key
       (not (string? rsa-key))
       (every? #(and (string? %)
                     (seq %))
               [endpoint-url callback-url spa-identifier shared-secret])))

(defn sources-authentication
  [{:keys [endpoint-url] :as config} tags history?]
  {:pre [(valid-config? config)]}
  (-> (oauth/authorization-request config)
      (request/json-body {:request_authorization
                          (merge {:tags tags}
                                 (when history?
                                   {:with_history history?}))})))

(defn sources
  [{:keys [endpoint-url] :as config} access-token]
  {:pre [(valid-config? config)]}
  (-> (oauth/authorized-request config access-token :get (str endpoint-url "/spa_api/sources"))
      request/accept-json))

(defn subscribe
  [{:keys [endpoint-url] :as config} access-token sources]
  {:pre [(valid-config? config)]}
  (-> (oauth/authorized-request config access-token :post (str endpoint-url "/spa_api/subscription_request"))
      (request/json-body {:requested_sources sources})))

(defn subscriptions
  [{:keys [endpoint-url] :as config} access-token]
  {:pre [(valid-config? config)]}
  (-> (oauth/authorized-request config access-token :get (str endpoint-url "/spa_api/subscriptions"))
      request/accept-json))

(defn batch-data-request
  [{:keys [spa-identifier rsa-key endpoint-url] :as config} subscription-identifiers from to]
  {:pre [(valid-config? config)]}
  (-> (request/request :post (str endpoint-url "/spa_api/data_requests"))
      (request/accept-json)
      (request/json-body
       {:data_request
        {:subscription_identifiers subscription-identifiers
         :from from
         :to to
         :neartime false}})
      (request/authenticate spa-identifier rsa-key)))

(defn batch-data-download
  [{:keys [spa-identifier rsa-key endpoint-url] :as config} request-id]
  {:pre [(valid-config? config)]}
  (-> (request/request :get (str endpoint-url "/spa_api/data_requests/" request-id "/data"))
      (request/accept-json)
      (request/authenticate spa-identifier rsa-key)))

(defn neartime-data-request
  [{:keys [spa-identifier rsa-key endpoint-url] :as config} subscription-identifiers interval]
  {:pre [(valid-config? config)]}
  (-> (request/request :post (str endpoint-url "/spa_api/data_requests"))
      (request/accept-json)
      (request/json-body
       {:data_request
        {:subscription_identifiers subscription-identifiers
         :interval (name interval)
         :neartime true}})
      (request/authenticate spa-identifier rsa-key)))

(defn remove-data-request
  [{:keys [spa-identifier rsa-key endpoint-url] :as config} request-id]
  {:pre [(valid-config? config)]}
  (-> (request/request :delete (str endpoint-url "/spa_api/data_requests/" request-id))
      (request/authenticate spa-identifier rsa-key)))
