(ns org.hellodata.api-client
  (:require [oauth.client :as oauth]
            [oauth.signature :as sig]
            [clojure.string :as string]
            [org.hellodata.api.request :as request]
            [clj-http.client :as http-client]
            [cheshire.core :as json]))

(defn new-consumer
  [base-url callback-url app-key shared-secret rsa-key]
  {:pre [base-url callback-url app-key shared-secret rsa-key]}
  (let [base-url (string/replace base-url #"/+$" "")]
    (-> (oauth/make-consumer app-key shared-secret
                             (str base-url "/oauth/request_token")
                             (str base-url "/oauth/access_token")
                             (str base-url "/oauth/authorize")
                             :hmac-sha1)
        (assoc ::base-url base-url
               ::app-key app-key
               ::rsa-key rsa-key
               ::callback-url callback-url))))

(defn new-oauth-request
  [consumer uri]
  (let [unsigned-params (-> (sig/oauth-params consumer
                                              (sig/rand-str 30)
                                              (sig/msecs->secs (System/currentTimeMillis)))
                            (assoc :oauth_callback (::callback-url consumer)))]
    (-> (oauth/build-oauth-token-request consumer
                                         uri
                                         unsigned-params
                                         nil)
        (assoc :url uri))))

(defn oauth-token?
  [{:keys [oauth_token oauth_token_secret]}]
  (boolean (and oauth_token oauth_token_secret)))

(defn request-sources-auth
  ([consumer tags]
   (request-sources-auth consumer tags nil))
  ([consumer tags history?]
   {:post [(oauth-token? %)]}
   (oauth/post-request-body-decoded (:request-uri consumer)
                                    (-> (new-oauth-request consumer (:request-uri consumer))
                                        (request/json-body {:request_authorization
                                                            (merge {:tags tags}
                                                                   (when history?
                                                                     {:with_history history?}))})))))

(defn auth-approval-url
  "The url for the user to approve access to requested authorization"
  [consumer request-token]
  {:pre [request-token]}
  (->> request-token
       :oauth_token
       (oauth/user-approval-uri consumer)))

(defn get-access-token
  "Get access token for user-approved request-token. `verifier` is the value
  that was provided by the redirect url by the user's browser after
  clicking though the `auth-approval-url`"
  [consumer request-token verifier]
  (oauth/access-token consumer request-token verifier))

(defn credentials
  [consumer token request-method url & params]
  (apply oauth/credentials consumer
         (:oauth_token token)
         (:oauth_token_secret token)
         request-method
         url
         params))

(defn signed-request
  "Build an ring-formatted oauth request ready for use by clj-http.client"
  [consumer token request-method url]
  (-> (credentials consumer token request-method url)
      (request/accept-api-v1)
      oauth/build-request
      (assoc :url url
             :request-method request-method)))

(defn sources-request
  "ring request to fetch sources data"
  [consumer access-token]
  (-> (signed-request consumer access-token :get (str (::base-url consumer) "/spa_api/sources"))
      request/accept-json))

(defn get-sources
  [consumer access-token]
  (-> (sources-request consumer access-token)
      http-client/request
      :body))

(defn subscribe-request
  [consumer access-token sources]
  (-> (signed-request consumer access-token :post (str (::base-url consumer) "/spa_api/subscription_request"))
      (request/json-body {:requested_sources sources})))

(defn post-subscriptions
  [consumer access-token sources]
  (-> (subscribe-request consumer access-token sources)
      http-client/request))


(defn subscriptions-request
  [consumer access-token]
  (-> (signed-request consumer access-token :get (str (::base-url consumer) "/spa_api/subscriptions"))
      request/accept-json))

(defn get-subscriptions
  [consumer access-token]
  (:body (http-client/request (subscriptions-request consumer access-token))))

(defn data-request
  [consumer subscription-identifiers from to neartime?]
  (-> (request/request :post (str (::base-url consumer) "/spa_api/data_requests"))
      (request/accept-json)
      (request/json-body
       {:data_request
        {:subscription_identifiers subscription-identifiers
         :from from
         :to to
         :neartime neartime?}})
      (request/authenticate (::app-key consumer) (::rsa-key consumer))))

(defn post-data-request
  [consumer subscription-identifiers from to neartime?]
  (-> (data-request consumer subscription-identifiers from to neartime?)
      http-client/request))
