(ns org.hellodata.api.request
  "Functions that build (but not send) HTTP requests."
  (:require [cheshire.core :as json]
            [clojure.string :as string]
            [org.hellodata.crypto :as crypto]))

(defn- request-method-str
  [{:keys [request-method method] :as request}]
  {:pre [(or request-method method)]}
  (string/upper-case (name (or request-method method))))

(defn- request-path
  [{:keys [url uri] :as request}]
  {:pre [(or url uri)]}
  (or uri (.getPath (java.net.URL. url))))

(defn sign
  "Sign a ring or http-kit style request using `rsa-key`.
  See `org.hellodata.crypto/private-key`"
  [{:keys [url body] :as request} rsa-key]
  (assoc-in request [:headers "X-Auth-Sig"]
            (crypto/sign-string rsa-key
                                (str (request-method-str request)
                                     (request-path request)
                                     (if body body "")))))

(defn json-body
  "Assoc `data` as json encoded request body and setup headers"
  [request data]
  (-> request
      (assoc-in [:headers "Content-Type"] "application/json")
      (assoc :body (json/generate-string data))))

(defn accept
  "Add accept-term to accept header. Creates accept header if
  there isn't one already."
  [request accept-term]
  (update-in request [:headers "Accept"]
             #(if % (str % ", " accept-term) accept-term)))

(defn accept-json
  "Setup headers and config for request to return an automatically
  decoded json response"
  [request]
  (-> request
      (assoc :as :json)
      (accept "application/json")))

(defn accept-api-v1
  "Specify HelloData.org SPA API version 1"
  ([request]
   (accept request "application/vnd.hd.spa-v1"))
  ([]
   (accept-api-v1 {})))

