(ns dead2
  #?(:clj
     (:import [java.nio ByteBuffer]
              [java.util Arrays]))
  (:require [criterium.core :as c]))

(set! *warn-on-reflection* true)

;;;; ============================================================================================
;;;; Misc util

(defn bimap [items]
  (->> items
       (map-indexed #(vector (- %1 32768) %2))
       (mapcat (fn [[a b]] [[a b] [b a]]))
       (into {})))



;;;; ============================================================================================
;;;; Binary util

(defn short->bytes [x]
  (#?(:clj  (partial into-array Byte/TYPE)
      :cljs clj->js)
    [(-> x (bit-shift-right 8) (bit-and 0xFF) (unchecked-byte))
     (-> x (bit-and 0xFF) (unchecked-byte))]))

;;;; ============================================================================================
;;;; Bitbuffer

(defn bit-write-buffer [n]
  {:array      #?(:clj  (byte-array n)
                  :cljs (js/Int8Array. n))
   :index-atom (atom 0)})

(defn put-bit! [{:keys [array index-atom]} bit]
  (if bit
    (let [index      @index-atom
          byte-index (quot index 8)
          bit-index  (rem index 8)]
      (#?(:clj aset-byte :cljs aset)
        array byte-index (-> array
                             (aget byte-index)
                             (bit-set bit-index)
                             (unchecked-byte)))
      (swap! index-atom inc))))

(defn bit-read-buffer [byte-array]
  {:array      byte-array
   :index-atom (atom 0)})

(defn take-bit! [{:keys [array index-atom]}]
  (let [index      @index-atom
        byte-index (quot index 8)
        bit-index  (rem index 8)
        bit        (-> array (aget byte-index) (bit-test bit-index))]
    (swap! index-atom inc)
    bit))

(defn reset-bit-buffer [{:keys [index-atom]}]
  (reset! index-atom 0))

;;;; ============================================================================================
;;;; Protocols

(defprotocol AutoByteBuffer
  (put! [this data type])
  (take! [this type])
  (reset! [this]))

(defprotocol AutoByteBufferPosition
  (position [this])
  (alter-position [this amount]))

;;;; ============================================================================================
;;;; Buffer implementation

(do #?(:clj  (extend-type ByteBuffer
               AutoByteBuffer
               (take! [this type]
                 (case type
                   ::byte (.get this)
                   ::short (.getShort this)
                   ::int (.getInt this)
                   ::float (.getFloat this)
                   ::double (.getDouble this)
                   ::char (.getChar this)
                   nil))
               (put! [this data type]
                 (case type
                   ::byte (.put this (byte data))
                   ::short (.putShort this (short data))
                   ::int (.putInt this (int data))
                   ::float (.putFloat this (float data))
                   ::double (.putDouble this (double data))
                   ::char (.putChar this (char data))
                   nil)
                 this)
               (reset! [this]
                 (.position this 0)))

       :cljs (extend-type js/DataView
               AutoByteBufferPosition
               (position [this]
                 (or (aget this "position") 0))
               (alter-position [this amount]
                 (let [value (position this)]
                   (aset this "position" (+ value amount))
                   value))

               AutoByteBuffer
               (take! [this type]
                 (case type
                   ::byte (.getInt8 this (alter-position this -1))
                   ::short (.getInt16 this (alter-position this -2))
                   ::int (.getInt32 this (alter-position this -4))
                   ::float (.getFloat32 this (alter-position this -4))
                   ::double (.getFloat64 this (alter-position this -8))
                   ::char (.getUint8 this (alter-position this -2))
                   nil))
               (put! [this data type]
                 (case type
                   ::byte (.setInt8 this (alter-position this 1) data)
                   ::short (.setInt16 this (alter-position this 2) data)
                   ::int (.setInt32 this (alter-position this 4) data)
                   ::float (.setFloat32 this (alter-position this 4) data)
                   ::double (.setFloat64 this (alter-position this 8) data)
                   ::char (.setUint8 this (alter-position this 2) data)
                   nil)
                 this)
               (reset! [this]
                 (aset this "position" 0)))))

(defn make-buffer [bits-length bytes-length]
  (let [length (+ 2 bits-length bytes-length)]
    {:buffer        #?(:clj  (byte-array length)
                       :cljs (js/ArrayBuffer. length))
     :bit-position  16
     :byte-position (+ 2 bits-length)}))

(defn wrap-bytes [bytes]
  (let [buffer #?(:clj (ByteBuffer/wrap bytes)
                  :cljs (js/DataView. bytes))]
    {:buffer        buffer
     :bit-position  16
     :byte-position (+ 2 (take! buffer ::short))}))

(defn unwrap-buffers [{:keys [bits bytes]}]
  #?(:clj  (let [bit-array    (:array bits)
                 bits-length  (count bit-array)
                 bytes-length (.position bytes)
                 merged       (byte-array (+ 2 bits-length bytes-length))]
             (System/arraycopy (short->bytes bits-length) 0 merged 0 2)
             (System/arraycopy bit-array 0 merged 2 bits-length)
             (System/arraycopy (.array bytes) 0 merged (+ 2 bits-length) bytes-length)
             merged)
     :cljs (let [bit-array    (->> bits (bits->bytes) (clj->js)) ; todo replace
                 bits-length  (count bit-array)
                 bytes-length (.-byteLength bytes)
                 merged       (js/Uint8Array. (+ 2 bits-length bytes-length))]
             (.set merged (short->bytes bits-length) 0)
             (.set merged bit-array 2)
             (.set merged (js/Uint8Array. bytes) (+ 2 bits-length))
             (.-buffer merged))))

;;;; ============================================================================================
;;;; De/serialization

(declare serialize-1)
(declare deserialize-1)

(def primitives
  #{::byte ::short ::int ::float ::double ::char})

(defn primitive? [type]
  (primitives type))

(defn serialize-primitive [schema _ buffer data]
  `(put! ~buffer ~data ~schema))

(defn deserialize-primitive [schema _ buffer]
  `(take! ~buffer ~schema))


(defn serialize-coll [[_ sub-schema] config buffer data]
  (let [data-piece (gensym "data-piece_")]
    `(do (put! ~buffer (count ~data) ::short)
         (run! (fn [~data-piece]
                 ~(serialize-1 sub-schema config buffer data-piece))
               ~data))))

(defn deserialize-coll [[coll-type sub-schema] config buffer]
  (let [f (case coll-type
            ::vector vec
            ::list seq
            ::set set)]
    `(~f (for [_ (range (take! ~buffer ::short))]
           (deserialize-1 ~sub-schema ~config ~buffer)))))


(defn serialize-tuple [schema config buffer data]
  (let [data-vec (gensym "data-vec_")]
    `(let [~data-vec (vec ~data)]
       ~@(map (fn [[index sub-schema]]
                (serialize-1 sub-schema config buffer `(get ~data-vec ~index)))
              (map-indexed vector schema)))))

(defn deserialize-tuple [schema config buffer]
  `(vector ~@(for [sub-schema (rest schema)]
               (deserialize-1 sub-schema config buffer))))


(defn serialize-keyword [_ config buffer data]
  (put! buffer (get-in config [:keyword-map data]) ::short))

(defn deserialize-keyword [_ config buffer]
  (get-in config [:keyword-map (take! buffer ::short)]))


(defn serialize-sub-schema [schema config buffer data]
  (serialize-1 (get-in config [:schemas schema]) config buffer data))

(defn deserialize-sub-schema [schema config buffer]
  (deserialize-1 (get-in config [:schemas schema]) config buffer))


(defn serialize-map [schema config buffer data]
  nil)

(defn deserialize-map [schema config buffer]
  nil)


(defn serialize-record [schema config buffer data]
  nil)

(defn deserialize-record [schema config buffer]
  nil)


; todo keyword-str, string, map, record, boolean, delta, bitbuffer as an big array, filled? -> arraycopy/slice





(defn serialize-1 [schema config buffer data]
  (cond
    (primitive? schema) (serialize-primitive schema config buffer data)
    :else nil))

(defn deserialize-1 [schema config buffer]
  (cond
    (primitive? schema) (deserialize-primitive schema config buffer)
    :else nil))

;;;; ============================================================================================
;;;; Public API

(defmacro defconfig
  "Defines a config with name @name, schemas @schemas
  and a list of optional arguments @args."
  [name schemas & args]
  nil)

(defn serialize
  "Serialize given @data as @schema, according to settings defined in @config."
  [data schema config]
  nil)

(defn deserialize
  "Deserialize the contents of given @buffer, according to settings defined in @config."
  [buffer config]
  nil)

;;;; ============================================================================================
;;;; Playground

; SCHEMA CONFIG BUFFER DATA
; config: :schemas, :keyword-map

(let [bits (repeatedly 1000 #(zero? (rand-int 2)))]
  (c/with-progress-reporting
    (c/quick-bench (->> bits (bits->bytes) (into-array Byte/TYPE)))))

(let [bits (byte-array 1000)]
  (c/with-progress-reporting
    (c/quick-bench (Arrays/copyOf bits 125))))

(def cc #(c/with-progress-reporting (c/quick-bench %)))

(let [bits   {:array (byte-array 1000) :index 30}
      put-fn (fn [{:keys [array index] :as bits} b]
               )]
  (cc (put-fn bits true)))

(cc (swap! (atom 0) inc))

;;;; ============================================================================================
;;;; Spec

(comment
  {buffers     {:bits  bit-buffer
                :bytes byte-buffer}

   bit-buffer  {:array      array
                :index-atom index-atom}

   byte-buffer {:clj  ByteBuffer
                :cljs DataView}

   raw-format  {:clj  byte-array
                :cljs ArrayBuffer}}

  {:clj {bit-buffer  {:array      byte-array
                      :index-atom (atom 0)}
         byte-buffer ByteBuffer
         "preallocate a big ByteBuffer and byte-array
         fill them with put! and take!
         "}})


