(ns leiningen.seria
  "Generates source code for seria builds."
  (:require [leiningen.core.main :as lein-main]
            [leiningen.core.eval :as lein-eval]
            [clojure.java.io :as io]))

(defn existing-file [file]
  (when-not (.exists file)
    (.mkdirs (.getParentFile file))
    (.createNewFile file))
  file)

(defn process-build [project build source target namespace pretty-print?]
  (let [source-file (io/file source)
        target-file (existing-file (io/file target))]
    (when-not (.exists source-file)
      (lein-main/abort (format "Source file does not exist: %s.\nAborted."
                               source)))
    (let [[command config-name & args] (read-string (slurp source-file))]
      (when-not (and (= 'defconfig command)
                     (symbol? config-name))
        (lein-main/abort (format "Invalid source file: %s.\nAborted."
                                 source)))
      (lein-main/info (format "Generating source code for build: %s ..."
                              build))
      (lein-eval/eval-in-project project
                                 `(spit ~(.getAbsolutePath target-file)
                                        (seria.prettify/ns-str
                                          '~(symbol namespace)
                                          '~config-name
                                          (seria.config/make-config ~@args)
                                          ~pretty-print?)
                                        :append false)
                                 '(require 'seria.config 'seria.prettify))
      (lein-main/info "Success!"))))

(defn seria
  "Generates source code for a given build specified in your project.clj."
  [project build]
  (let [{:keys [source target namespace pretty-print?] :or {pretty-print? false}} (get-in project [:seria build])]
    (when-not (and source target namespace)
      (lein-main/abort (format (str "Invalid build: %s. Please specify your :source, :target "
                                    "and :namespace parameters.\nAborted.")
                               build)))
    (try
      (process-build project build source target namespace pretty-print?)
      (catch Exception e
        (lein-main/abort (format "Failed to process build: %s.\nUnexpected exception: %s.\nAborted."
                                 build
                                 (.getMessage e)))))))
