(ns morri.meth450k.common.db-reporting
  (:require [morri.meth450k.common
             [utils :as utils :refer [tprn]]
             [db-utils :as db-utils]]
            [clojure.java.jdbc :as jdbc]
            [java-jdbc.ddl :as ddl]
            [honeysql.core :as sql]
            [honeysql.helpers :refer :all]
            [clojure.walk :as walk]))

(defn cols-from-template [template]
  ((comp distinct flatten) (map :db-cols template)))

(defn make-sql-query [table template limit? index-col]
  (let [cols (cols-from-template template)]
    (if limit?
      (-> (apply select cols)
          (from table)
          (where [:= index-col :?indx]))
      (-> (apply select cols)
          (from table)))))

;; (defn setup-limit-probes [db table limit-probes index-col]
;;   (do
;;     (db-utils/try-drop-table db :limit_table)
;;     (when limit-probes
;;       (jdbc/db-do-commands db (tprn (ddl/create-table
;;                                      :limit_table
;;                                      [:limit_probe_number "integer"]
;;                                      [:limit_probe "varchar(40)"])))
;;       (apply jdbc/insert! db :limit_table [:limit_probe_number :limit_probe]
;;              (map vector (iterate inc 1) limit-probes))
;;       (db-utils/try-create-index db :limit_table [:limit_probe])
;;       (db-utils/try-create-index db :limit_table [:limit_probe_number])
;;       (db-utils/try-create-index db :limit_table [:limit_probe :limit_probe_number])
;;       (db-utils/try-create-index db table (vector index-col)))))

(defn apply-template-to-row
  "Apply template to res. When the templated data is invalid return
nil for the individual data point"
  [template res]
  (when res
    (for [{:keys [col db-cols transform-fn valid?]} template
          :let [data (transform-fn (walk/postwalk-replace res db-cols))]]
      (when (valid? data) data))))

(defn make-explain-sql [sql]
  (let [explain-query [(str "explain " (first sql))]
        parameters (next sql)]
    (tprn (if parameters (conj explain-query parameters) explain-query))))

(defn db-report
  "Returns a lazy sequence obtained by applying template to table in
  db.  Optionally limit the results to limit-probes which can be
  looked up by index-col in db.table"
  ([db table template]
     (db-report db table template false false))
  ([db table template limit-probes index-col]
     (let [sql-query (make-sql-query table template limit-probes index-col)
           db-results (if limit-probes
                        (do
                          (db-utils/try-create-index db table (vector index-col))
                          (for [a-probe limit-probes]
                            (first (jdbc/query db
                                               (sql/format
                                                sql-query
                                                :params {:indx a-probe})
                                               :identifiers identity))))
                        (jdbc/query db
                                    (sql/format sql-query)
                                    :identifiers identity))]
       (map (partial apply-template-to-row template) db-results))))
