(ns morri.meth450k.common.command-line
  (:require clojure.tools.cli
            [clojure.string :as string]))

(defn do-not-read-symbol [s]
  (let [after-read (read-string s)]
    (if-not (symbol? after-read) after-read s)))

(defn validate-list
  "Returns a function to validate a list of command line inputs using valid? Always returns a list even when presented with a single argument"
  [valid?]
  (fn [args]
    (let [input-list (map do-not-read-symbol (string/split args #"\s+"))]
      (if (every? valid? input-list)
        input-list
        (throw (Exception.
                (str \" (clojure.string/join \space (remove valid? input-list)) \" 
                     " are not valid " valid? " is required.")))))))

(defn validate
  "Returns a function to validate command line inputs using valid?
  predicate. Keywords and numbers are read in, but strings remain as
  strings" 
  [valid?]
  (fn [arg]
    (let [after-read (do-not-read-symbol arg)]
      (if (valid? after-read)
        after-read
        (throw (Exception.
                (str \" after-read \" 
                     " is not valid " valid? " is required.")))))))

((validate-list #{:one :two :three}) ":one :two :three")

((validate-list #{:one :two :three}) ":one")

((validate #{:one :two :three}) ":one")

((validate-list #{1 2 3}) "1 2 3")

;; ((validate #{"one" "two" "three"}) "one two three")

(defn ensure-int [arg-in]
  (let [arg (read-string arg-in)]
    (if (integer? arg)
      arg
      (throw (Exception. (str "\"" arg "\" is not an integer."))))))

(defn parse-command-line [args opts-config]
  (let [[options args banner] (apply clojure.tools.cli/cli args opts-config)]
    (when (:help options)
      (println banner) (System/exit 0))
    (prn options)
    options))
