(ns morri.lib.binning)

(def not-neg? (complement neg?))

(defn tprn [x] (prn x) x)

(defn which-bin
  "Returns a function for assigning a number x to a bin. The function
  returns the bin center for any value x.  The binning is based on
  (quot x bin-width), so with bin-width 1 we get bins (-4, -2] (-2,0)
  [0,2) [2,4) where \"(\" and \")\" mean an open endpoint and \"[\"
  and \"]\" mean a closed endpoint.

  Note the asymetric behaviour around zero.  The default is to give 0
  to the pos interval making (-2,0) and [0,2).  This can be switched
  by setting :zero-is-pos? to false to generate (-2,0] (0,2).

  If start is provided, then the binning described above is offset so
  that start is considered zero.

  Default is to return the center of the chosen bin, but this can be
  changed by passing :return as :lower-limit or :upper-limit
  "

  [bin-width & {:keys [zero-is-pos? start return]
               :or {zero-is-pos? true
                    start 0
                    return :bin-center}}]
  (fn [x]
    (let [bin-center (/ bin-width 2)
          offset-x (- x start)
          bin-base (* bin-width (quot offset-x bin-width))
          pos-pred? (if zero-is-pos? not-neg? pos?)
          bin-pre-offset (if (pos-pred? offset-x)
                           (+ bin-base bin-center)
                           (- bin-base bin-center))]
      (cond (= :bin-center return)
            (+ bin-pre-offset start)
            (= :lower-limit return)
            (+ bin-base start)
            (= :upper-limit return)
            (+ bin-base start bin-width)
            :else (throw (Exception.
                          (str return " not supported for :return")))))))

(defn range-with-stepsize [start stop step]
  (map #(* step %) (range (/ start step) (inc (/ stop step)))))

;; Better might be to have which bin return a bin number, and have
;; another function return a map from bin numbers to [lower-limit
;; bin-center upper-limit]

;; Negative number mean bins before start, positive numbers mean bins > start.

;; ---------------------------------------------------------------------------
;; ***************************************************************************
;; ---------------------------------------------------------------------------

;; New binning API.

(defn bin-dispatch
  ([bin-opts val] (:binning-type bin-opts))
  ([start width id-or-val] :low-level))       ;Use :low-level once we
                                              ;already have
                                              ;start and width

(defmulti bin-id bin-dispatch)
(defmulti bin-center bin-dispatch)
(defmulti bin-lower bin-dispatch)
(defmulti bin-upper bin-dispatch)

;; ---------------------------------------------------------------------------
;; :low-level binning
;; Internal functions to use once we've checked for the edges and
;; computed the bin width if needed.

(defmethod bin-id :low-level
  [start width val]
  (let [offset-val (- val start)
        raw-id (int (quot offset-val width))]
    ;; Don't let the zero bin be double sized by making the -1
    ;; bin [(start - width) start)
    (if (neg? offset-val) (dec raw-id) raw-id)))

(defn lower-edge [start width id]
  (+ start (* id width)))

(defmethod bin-lower :low-level 
  [start width id]
  (lower-edge start width id))

(defmethod bin-center :low-level 
  [start width id]
  (+ (lower-edge start width id) (/ width 2)))

(defmethod bin-upper :low-level
  [start width id]
  (+ (lower-edge start width id) width))

;; ---------------------------------------------------------------------------
;; start-width-binning
;; Use this type of binning when it is convenient to specify the start
;; of the binning and the width of the bins.

(defmethod bin-id :start-width-binning
  [{:keys [start width start-is-pos?]
    :or {start 0
         width 1
         start-is-pos? true}}
   val]
  (if (== val start)
    (if start-is-pos? 0 -1)
    (bin-id start width val)))

(defmethod bin-lower :start-width-binning
  [{:keys [start width] :or {start 0 width 1}} id]
  (bin-lower start width id))

(defmethod bin-center :start-width-binning
  [{:keys [start width] :or {start 0 width 1}} id]
  (bin-center start width id))

(defmethod bin-upper :start-width-binning
  [{:keys [start width] :or {start 0 width 1}} id]
  (bin-upper start width id))

;; ---------------------------------------------------------------------------
;; range-n-bins-binning
;; Use this type of binning when it is convenient to specify the
;; binning range and the number of bins.

(defn bin-width [start stop n-bins]
  (/ (- stop start) n-bins))

(defmethod bin-id :range-n-bins-binning
  [{:keys [start stop n-bins start-closed? stop-closed?]
    :or {start-closed? true stop-closed? true}}
   val]
  (cond
   (< val start) -1                     ;Push vals below start into
                                        ;the -1 bin
   (> val stop) n-bins                  ;Push vals above stop in the
                                        ;n-bins bin
   (== val start) (if start-closed? 0 -1)
   ;; first bin is 0, so nth bin is (dec n-bins)
   (== val stop) (if stop-closed? (dec n-bins) n-bins)
   :else (bin-id start (bin-width start stop n-bins) val)))

(defmethod bin-lower :range-n-bins-binning
  [{:keys [start stop n-bins]} id]
  (bin-lower start (bin-width start stop n-bins) id))

(defmethod bin-center :range-n-bins-binning
  [{:keys [start stop n-bins]} id]
  (bin-center start (bin-width start stop n-bins) id))

(defmethod bin-upper :range-n-bins-binning
  [{:keys [start stop n-bins]} id]
  (bin-upper start (bin-width start stop n-bins) id))


;; ---------------------------------------------------------------------------
;; ***************************************************************************
;; ---------------------------------------------------------------------------
;; 2-D Binning

(defn bin-id-2d [x-opts y-opts x y]
  [(bin-id x-opts x)
   (bin-id y-opts y)])

(defn bin-lower-2d [x-opts y-opts x y]
  [(bin-lower x-opts x)
   (bin-lower y-opts y)])

(defn bin-center-2d [x-opts y-opts x y]
  [(bin-center x-opts x)
   (bin-center y-opts y)])

(defn bin-upper-2d [x-opts y-opts x y]
  [(bin-upper x-opts x)
   (bin-upper y-opts y)])
