# Montoux Test

A collection of testing utilities for ClojureScript used by [Montoux](https://montoux.com).

## [`montoux.test.core`](src/montoux/test/core.cljs)

ClojureScript testing is more complicated than Clojure testing because, in general, macros are evaluated during
compilation on the server, and are not available in the browser. ClojureScript's `cljs.test` library works around
this problem by providing `cljs.test/run-all-tests` – a macro that generates ClojureScript that can be run in the
browser to run tests with the necessary bindings in place for reporting. This macro doesn't allow dynamic changes to the
test namespaces because its regex-based filtering takes place during compilation.

`montoux.test.core` provides `deftests` and `run-tests` which, together, allow you to generate code to run your tests at
compile time, but decide which tests to run at runtime. This facilites controlling which test namespaces are run by
interacting with our browser-based UI.

## [`montoux.test.html`](src/montoux/test/html.cljs)  

For browser-based testing, set up cljsbuild or figwheel to run an entry point that looks something like this:
 
```clojure
(ns main
  (:require [montoux.test.core :as core :include-macros true]
            [montoux.test.html :as html]
            ;; include all your test namespaces here
            ))

;; enable writing test output to js console
(enable-console-print!)

;; generates a test factory that can be passed to `run-tests`
(core/deftests tests)

;; every call to this function will re-run tests and re-render UI output
(defn run-tests []
  (core/run-tests tests (html/initialise (core/default-env))))

;; hook into browser history so that links can re-run tests
(html/on-url-change #'run-tests)

;; run tests for the first time
(run-tests)
```

When loaded by the browser, your tests will run automatically when figwheel updates or when you navigate using one of
the links on the page. This helps support fast test-driven development.

![Example](/docs/report.png)

The HTML output provides the following features:

 * run only one namespace by clicking on the specific namespace link
 * run all tests by clicking on the summary link (top bar)
 * show/hide a list of tests by clicking anywhere else on the namespace row
 * show/hide the output from a single test by clicking on the test row (not the link)
 * click the small link to the right of a test failure to jump to that source location ([Cursive](https://cursive-ide.com/))
 * test output is also available in the Javascript Console

## [`montoux.test.node`](src/montoux/test/node.cljs)

This namespace provides utilities for setting the exit code based on test output in nodejs.
See [`test.node`](/test/test/node.cljs). 

## [`montoux.test.mock`](src/montoux/test/mock.cljs)

We commonly make important API entry points dynamic to facilitate mock-based testing. The `mock` namespace provides
generator functions that produce stateful objects that behave like functions and keep track of calls. For example:

```clojure
(deftest test-important-api
  (binding [backend/store! (mock/stub :ok)]
    (is (= :ok (important-api :value)))
    (is (= [:value]
           (mock/last-call backend/store!)))
    ))
``` 

## Installation

Montoux Test is available from [clojars.org](https://clojars.org/montoux/test).

Add this to your `:dependencies` in `project.clj`:

    [montoux/test "1.0.0"]

## Montoux

[Montoux](http://montoux.com) is the global leader in pricing transformation for the life insurance industry.
Our customers include several of the world's leading insurance providers and we are expanding our business in
the United States and Asia, as well as Australia and New Zealand.  

## License

Copyright © 2018 [Montoux Limited](https://montoux.com)

Distributed under the Eclipse Public License either version 1.0 or (at your option) any later version.
