(ns radix.ring-metrics
  (:require [clojure.string :as str]
            [metrics.timers :refer [time! timer]])
  (:import [java.util.concurrent TimeUnit]
           [java.util.regex Pattern]))

;; When collecting metrics for resources in an application, it's not sensible to
;; collect values that change on every request, for example if the path for a
;; particular request contains user ids or some other attribute with a range of
;; values. So, the purpose of this namespace is to allow such values to be grouped.
;; The middleware function `wrap-per-resource-metrics` allows aggregation for metrics
;; so that a resource like `/{territory}/users/{a guid}/credit` doesn't produce
;; thousands of metrics paths one for each combination of territory and guid; instead
;; any calls to this resource will result in metrics on the path `/TERRITORIES/users/GUID/credit`.
;;
;; There are some default replacements in this namespace. If you want others, you
;; can just create your own and call the customisable version of `wrap-pre-resource-metrics`.

(def replace-guid
  [#"[0-9a-fA-F]{8}-[0-9a-fA-F]{4}-[0-9a-fA-F]{4}-[0-9a-fA-F]{4}-[0-9a-fA-F]{12}" "GUID"])

(def replace-mongoid
  [#"[0-9a-fA-F]{24}" "MONGOID"])

(def replace-number
  [#"[0-9][0-9]+" "NUMBER"])

(def replace-territory
  [#"/[a-z]{2}/" "/TERRITORIES/"])

(defn replace-outside-app
  "Create an aggregation to collect all requests for resources outside
  the application's context path into one big target called
  'OTHER'. Useful for hiding the vast set of nonsense requests created
  by security scans, and any other irrelevant 404s generated by clients
  making calls outside the application."
  [context-path]
  [(re-pattern (str "^[A-Z]*\\.(?!" (Pattern/quote context-path) ").*$")) "OTHER"])

(defn- apply-regex
  [path [pattern replacement]]
  (str/replace path pattern replacement))

(defn- apply-aggregations
  [path aggregations]
  (reduce apply-regex path aggregations))

(defn clean-metric-name
  [name]
  (-> name
      (str/replace " " ".")
      (str/replace "./" ".")
      (str/replace "/" ".")
      (str/replace #"\p{Cntrl}" "")
      (str/replace #"^\.+" "")
      (str/replace #"\.+$" "")))

(defn metric-name
  [request response aggregations & [creator-fn]]
  (let [name (-> (str/upper-case (name (:request-method request)))
                 (str "." (:uri request))
                 (apply-aggregations aggregations)
                 clean-metric-name
                 (str "." (:status response)))]
    (if creator-fn
      (creator-fn name)
      ["info" "resources" name])))

(defn wrap-per-resource-metrics
  "A middleware function to add metrics for all routes in the given
  handler. The simpler form adds default aggregators that replace GUIDs,
  Mongo IDs and Numbers with the constants GUID, MONGOID and NUMBER so
  that metric paths are sensible limited. Use the second form to specify
  your own replacements."
  ([handler]
     (wrap-per-resource-metrics handler [replace-guid replace-mongoid replace-number]))
  ([handler aggregations]
     (fn [request]
       (let [start (System/currentTimeMillis)
             response (handler request)
             duration (- (System/currentTimeMillis) start)
             timer (timer (metric-name request response aggregations))]
         (.update timer duration TimeUnit/MILLISECONDS)
         response))))
