(ns milesian.sequence-diagram
  (:require
   [clojure.walk :refer (postwalk)]
   [org.httpkit.client :refer (request) :rename {request http-request}]
   [com.stuartsierra.component :as component]
   [cheshire.core :refer (encode decode)]
   [clojure.string :as st]
   [clojure.pprint :refer (pprint)]
   [camel-snake-kebab :refer (->kebab-case-keyword ->camelCaseString)]
   [milesian.aop.utils  :refer (extract-data)]
            ))

(defn process-maps [fm t]
  (postwalk (fn [fm]
              (cond
               (map? fm) (reduce-kv (fn [acc k v] (assoc acc (t k) v)) {} fm)
               :otherwise fm)) fm))

(defn ->clj
  "Convert JSON keys into Clojure keywords. This is because we receive
  JSON but want to process it as Clojure."
  [fm]
  (process-maps fm ->kebab-case-keyword))

(defn ->js
  "Convert Clojure keywords into JSON keys. This is because we respond
  with JSON."
  [fm]
  (process-maps fm ->camelCaseString))

(defn request [method uri & {:keys [data]}]
  @(http-request
          (merge
           {:method method
            :url uri
            :headers
            (merge
             {"Content-Type" "application/json"
              "Accept" "application/json"})}

           (when data {:body (str (encode (->js data)))}))
          identity))

(defn replace- [s]
  (clojure.string/replace s #"-" "_"))


(defn function-invocation
  [*fn* this args]
  (let [{:keys [id who fn-name fn-args]} (extract-data *fn* this args)]
    (format "%s->%s: %s %s" (replace- who) (replace- id) fn-name fn-args)))

(defn function-return
  [*fn* this args]
  (let [{:keys [id who fn-name fn-args]} (extract-data *fn* this args)]
    (format "%s->%s:" (replace- id) (replace- who))))

(def example-seq  "a->b:....\n c->d:....\n e->e:....")

#_(reduce #(str % (highlight-node %2)) ""
          )

(defn extract-seq-nodes [m]
  (into #{}
       (flatten
        (map #(-> (st/split %  #":")
                  (first )
                  (st/split  #"->"))
             (st/split m  #"\n")))))
(def store (atom []))

(defn store-message [m k]
  (swap! store conj [m k]))
(def port 8011)
(def url-seq  (format "http://localhost:%s/publish-sequence" port))
(def url-graph  (format "http://localhost:%s/publish-graph" port))

;;example sequence: "Alice->Bob: Hello Bob, how are you? \n Note right of Bob: Bob thinks \n Bob-->Alice: I am good thanks!"
(defn publish-sequence [m]
  (request :post url-seq :data {:sequence m}))


(defn highlight-node [node]
  (str node " [style=\"fill: #f77; font-weight: bold\"]; ")
  )

(defn graph-dagree [highlight-nodes system-graph]
  (str "digraph {node [rx=5 ry=5 labelStyle=\"font: 300 14px Helvetica\"]; edge [labelStyle=\"font: 300 14px Helvetica\"]; "
       (reduce #(str % (highlight-node %2)) "" highlight-nodes)
      (->> (reduce-kv (fn [c k v]
                        (apply conj c (mapv #(str (replace- (name k)) " -> " (replace- (name %))) v))
                        ) [] system-graph)
           (clojure.string/join "; " )
           )
      "}"))


;;example graph format: "digraph {A -> B -> C; B -> D; D -> E;}"
;    node [rx=5 ry=5 labelStyle=\"font: 300 14px 'Helvetica Neue', Helvetica\"]
;    edge [labelStyle=\"font: 300 14px 'Helvetica Neue', Helvetica\"]
;; E [style="fill: #f77; font-weight: bold"];
(defn publish-graph [m]
  (request :post url-graph :data {:graph m}))

(defn graph-system [system] (reduce-kv (fn [c k v] (assoc c k (last (component/dependencies v)))) {} system))

(defn try-to-publish [s system]
  (let [closed (filter (fn[[_ k]] (= k :closed)) @s)]
    (when (= (count closed) (count (filter (fn[[_ k]] (= k :opened)) @s)))
      (let [m (reduce (fn [s [m _]] (str s m "\n")) "" @s)]
        (println m)
        (publish-sequence m)
        (publish-graph (graph-dagree (extract-seq-nodes m) (graph-system system)))
        )
      (reset! s []))))


(defn visualisation-invocation [system & opts]
  (fn [*fn* this & args]
   (store-message (function-invocation *fn* this args) :opened)
   (let [res (apply *fn* (conj args this))]
     (store-message (function-return *fn* this args) :closed)
     (try-to-publish store system)
     res)))
