(ns ^{:author "Mike Ananev, @MikeAnanev"}
  middlesphere.util
  (:require [middlesphere.log :as log])
  (:import (javax.crypto Cipher)
           (java.time.format DateTimeFormatter)
           (java.util Locale)
           (java.time LocalDateTime)))


(defn dev-mode?
  "# Checks if development mode is turned on.
     Dev mode is turned on/off by symbol `dev-mode` defined in namespace `user`

  * Returns:
  	  _true_  if development mode is turned on.
  	  _false_ if development mode is turned off."
  []
  (if-let [dm (resolve 'user/dev-mode)]
    (deref dm)
    false))


(defmacro dev-only
  "If 'development only' mode turned on, returns `body` wrapped in `do`.
  If 'development only' mode is off returns nil.
  By default dev-only mode is turned on, when running REPL (:repl alias)
  and turned off for other tasks."
  [& body]
  (when (true? (dev-mode?))
    (conj body 'do)))


(defn doc-template
  "# Generate doc string template for given `args` argument list.
   * Params:
  	  **`desc`**   - _String_ with description what function do.
  	  **`args`**   - _vector_ of symbols (argument names).
  	  **`return`** - _String_ with description what function return.
   * Returns:
  	  _String_ as doc string template."
  [desc args return]
  (let [max-len (if (empty? args) 0 (apply max (mapv count (mapv str args))))
        arg-str (if (zero? max-len) "  \t  no params. \n"
                                    (apply str (for [a args]
                                                 (format "  \t  **`%s`** %s- \n" (str a)
                                                         (apply str (repeat (- max-len (count (str a))) " "))))))]
    (format "# %s\n\n   * Params:\n%s\n   * Returns:\n  \t  %s" desc arg-str return)))


(defn global-exception-hook
  "# Sets global exception hook to catch any uncaught exceptions and log it.

   * Params:
  	  no params.

   * Returns:
  	  _nil_"
  []
  (Thread/setDefaultUncaughtExceptionHandler
    (reify Thread$UncaughtExceptionHandler
      (uncaughtException [_ th ex]
        (log/error :msg "got uncaught exception" :desc ex :thread (.getName th))))))


(defn ctrl-c-hook
  "# Set global hook to catch <ctrl-c> event, log it and execute given function,
  then shutdown Clojure agents.

   * Params:
  	  **`func-to-call`** - function, with no args, which should be executed during ctrl-c event.

   * Returns:
  	  _nil_"
  [func-to-call]
  (-> (Runtime/getRuntime)
      (.addShutdownHook (java.lang.Thread. (fn []
                                             (log/info :msg "got ctrl-c event!")
                                             (func-to-call)
                                             (shutdown-agents))))))

(defn jce-unlimited?
  "# Checks if JVM has unlimited cryptography strength.

   * Params:
  	  no params.

   * Returns:
  	  _true_ - JVM has unlimited cryptography strength.
     _false_ - JVM has unsecured cryptography (NOT FOR PRODUCTION)."
  []
  (= Integer/MAX_VALUE (Cipher/getMaxAllowedKeyLength "AES")))



(defmacro project-version
  "# Get current project version from Leiningen project using System properites.

   * Params:
  	  **`project-name`** - _String_ name of project without ns.

   * Example: (project-version \"util\")

   * Returns:
  	  _String_ - current version of project."
  [project-name]
  (System/getProperty (str project-name ".version")))


(defmacro project-build-time
  "# Get current date and time as string.

   * Params:
  	  no params.

   * Returns:
  	  _String_ - current date and time in format dd MMMM YYYY, HH:mm:ss."
  []
  (let [pattern (DateTimeFormatter/ofPattern "dd MMMM YYYY, HH:mm:ss")
        fmt     (.withLocale pattern (Locale. "en_US"))]
    (.format (LocalDateTime/now) fmt)))



(defmacro safe
  "# Execute any S-expressions inside try-catch block.

   * Params: any number of S-exp.

   * Returns:
  	  _value_ - any value if success.
                         _nil_    - if got Exception"
  [bindings? & forms]
  (let [bindings (when (and (even? (count bindings?)) (vector? bindings?))
                   bindings?)
        forms    (if bindings forms (cons bindings? forms))
        except   `(catch Exception e#)]
    (if bindings
      `(let ~bindings (try ~@forms ~except))
      `(try ~@forms ~except))))


(defmacro safe-log
  "# Execute any S-expressions inside try-catch block.

   * Params:
  	  **`forms`** - any number of S-exp.

   * Returns:
  	  _value_ - any value if success.
      _nil_    - if got Exception and log it."
  [& forms]
  `(try
     ~@forms
     (catch Exception e#
       (log/error :msg "Exception" :desc {:file    *file*
                                          :exp     (quote ~@forms)
                                          :src     ~(meta &form)
                                          :ex-type (type e#)
                                          :ex-msg  (.getMessage e#)}))))


(defmacro nif-let
  "# Nested if-let. If all bindings are non-nil or true, executes body in the context of those bindings.
  If a binding is nil or false, evaluates its `else-expr` form and stops there.
  `else-expr` is otherwise not evaluated.

   * Params:
  	  **`bindings`** - is two S-exp: binding-form else-expr

   * Returns:
  	  _value_ - any value which let block returns or `else-expr` where nil/false is occurred.

  Taken from here: https://www.proofbyexample.com/nested-if-let-in-clojure.html

  Example:
    (nif-let [a x :a-is-nil
              b y :b-is-nil
              c z :c-is-nil
              d w :d-is-nil]
            (+ a b c d))"
  [bindings & body]
  (cond
    (zero? (count bindings)) `(do ~@body)
    (symbol? (bindings 0)) `(if-let ~(subvec bindings 0 2)
                              (nif-let ~(subvec bindings 3) ~@body)
                              ~(bindings 2))
    :else (throw (IllegalArgumentException. "symbols only in bindings"))))


(defmacro if-let*
  "# Multiple binding version of if-let.

   * Params:
  	  **`bindings`** - normal let bindings.
  	  **`then`**     - S-form which will be evaluated if all bindings not nil.
  	  **`else`**     - S-form which will be evaluated if one of bindings is nil. If `else` is absent then return _nil_.

   * Returns:
  	  _value_ - any value which `then` or `else` produces.

  	  Example:
  	    (if-let* [a 1
                  b 2
                  c 3 ]
           (+ a b c)
           :some-has-nil)"
  ([bindings then]
   `(if-let* ~bindings ~then nil))
  ([bindings then else]
   (if (seq bindings)
     `(if-let [~(first bindings) ~(second bindings)]
        (if-let* ~(vec (drop 2 bindings)) ~then ~else)
        ~else)
     then)))


(defn obfuscate-map
  "obfuscate values in map with sensitive data on all levels.
  map must contain meta data with :secrets key and vector of keywords which should be obfuscated as value.
  Example: (with-meta {:abc 123 :pwd \"secret\" :eee {:ssn 12345}}
             {:secrets [:pwd :ssn]})
  return map of the same structure with obfuscated sensitive data on all levels.
  Example: {:abc 123 :pwd \"*obfuscated*\" :eee {:ssn 0}}"
  [obj]
  (reduce (fn [acc kw]
            (clojure.walk/postwalk
              (fn [obfusc-map]
                (if (and (map? obfusc-map) (kw obfusc-map))
                  (update-in obfusc-map [kw] (fn [x]
                                               (cond
                                                 (string? x)  "*obfuscated*"
                                                 (float? x)   0.0
                                                 (integer? x) 0
                                                 (decimal? x) 0.0M
                                                 (uuid? x)    #uuid "00000000-0000-0000-0000-000000000000"
                                                 (vector? x)  ["*obfuscated*"]
                                                 (map? x)     {:obfuscated "*obfuscated*"}
                                                 :else        "*obfuscated*")))
                  obfusc-map))
              acc))
          obj
          (first (vals (meta obj)))))