(ns om-stuff.ajax-utils
  (:refer-clojure :exclude [get])
  (:require-macros [cljs.core.async.macros :refer [go]])
  (:import  [goog History Uri]
            [goog.Uri QueryData])
  (:require [goog.dom.classlist :as classlist]
            [goog.events :as events]
            [goog.dom :as gdom]
            [goog.history.EventType :as EventType]
            [cljs.core.async :refer [put! <! chan]]
            [clojure.string :as string]
            [cljs.reader]
            [om.core :as om :include-macros true]
            [bidi.bidi :as bidi]
            [om-stuff.translations :refer [t]]
            [cljs-http.client :as http]))

(extend-protocol bidi/Matched
  cljs.core.PersistentHashMap
  (resolve-handler [this m] (some #(bidi/match-pair % m) this))
  (unresolve-handler [this m] (some #(bidi/unmatch-pair % m) this)))

(def ^:dynamic *routes*)

(defn path-for
  [& args]
  (apply bidi/path-for *routes* args))

(defn get-params []
  (let [qd (.getQueryData (Uri. (.-search js/location)))]
    (into {}
          (for [key (.getKeys qd)]
            [(keyword key) (.get qd key)]))))

(defn iframe?
  []
  (try
    (not= (.-self js/window) (.-top js/window))
    (catch js/Error e true)))

(defn browser-redirect
  [location url]
  (set! (.-href location) url))

(defn redirect
  "Redirect to a given url."
  ([url]
   (try
     (if (iframe?)
       (browser-redirect (.-top (.-location js/window)) url)
       (browser-redirect (.-location js/window) url))))
  ([url params] (redirect url params nil))
  ([url params anchor]
   (let [uri (Uri. url)
         query-data (QueryData.)]
     (doseq [[k v] params]
       (.add query-data (name k) v))
     (.setQueryData uri query-data)
     (redirect (str (.toString uri) anchor)))))

(defn redirect-absolute
  [& args]
  (str (path-for :home) (apply redirect args)))

(def default-media-params :edn-params)

(def error-chan (chan))

(defn status? [expected {:keys [status]}]
  (= expected status))

(defn ok?
  [{:keys [status]}]
  (and (>= status 200) (< status 300)))

(defn to-keyword [v]
  (if (coll? v)
    (t (or (first (filter keyword? v)) :invalid-value))    
    (t (keyword v))))

(defmulti transform-errors :status)

(defmethod transform-errors :default
  [response]
  response)

(defmethod transform-errors 400
  [response]
  (let [errors (into {} (map (fn [[k v]] [(keyword k) (to-keyword v)])
                             (get-in response [:body :schema-error])))]
    {:errors errors}))

(defmethod transform-errors 401
  [response]
  (redirect (str (path-for :login) (.-hash js/location))))

(defn stop-ajax-wait! []
  (classlist/remove (.. js/document (getElementById "ajax-container")) "loading"))

(defn start-ajax-wait! []
  (classlist/add (.. js/document (getElementById "ajax-container")) "loading"))

(defn ajax*
  [method path & [params options]]
  (let [bg? (:bg? options)
        c (method path
                  (-> (assoc options
                             default-media-params (dissoc params :errors))
                      (dissoc :bg?)))
        o (chan)]
    (if-not bg? (start-ajax-wait!))
    (go
      (let [{:keys [body] :as result} (<! c)]
        (if-not bg? (stop-ajax-wait!))
        (when-not (ok? result)
          (put! error-chan result))
        (put! o (transform-errors result))))
    o))

(def get    (partial ajax* http/get))
(def put    (partial ajax* http/put))
(def post   (partial ajax* http/post))
(def delete (partial ajax* http/delete))

(defn get-bg [path & [params options]]
  (ajax* http/get path params (assoc options :bg? true)))
