(ns ctrl.merge
  (:refer-clojure :exclude [merge])
  (:require [clojure.core :as c]
            [clojure.set :as set]))

(defn- meta* [obj]
  (if #?(:clj  (instance? clojure.lang.IObj obj)
         :cljs (satisfies? IMeta obj))
    (meta obj)))

(defn- with-meta* [obj m]
  (if #?(:clj  (instance? clojure.lang.IObj obj)
         :cljs (satisfies? IWithMeta obj))
    (with-meta obj m)
    obj))

(defn- displace? [obj]
  (-> obj meta* :displace))

(defn- replace? [obj]
  (-> obj meta* :replace))

(defn- top-displace? [obj]
  (-> obj meta* :top-displace))

(defn- different-priority? [left right]
  (boolean (or (some (some-fn nil? displace? replace?) [left right])
               (top-displace? left))))

(defn- remove-top-displace [obj {:keys [::replace-nil]}]
  (cond replace-nil nil
        (top-displace? obj) obj
        :else (vary-meta obj dissoc :top-displace)))

(defn- pick-prioritized [left right options]
  (cond (nil? left) right
        (nil? right) (remove-top-displace left options)

        (top-displace? left) right

        (and (displace? left) ;; Pick the rightmost
             (displace? right)) ;; if both are marked as displaceable
        (with-meta* right
                    (c/merge (meta* left) (meta* right)))

        (and (replace? left) ;; Pick the rightmost
             (replace? right)) ;; if both are marked as replaceable
        (with-meta* right
                    (c/merge (meta* left) (meta* right)))

        (or (displace? left)
            (replace? right))
        (with-meta* right
                    (c/merge (-> left meta* (dissoc :displace))
                             (-> right meta* (dissoc :replace))))

        (or (replace? left)
            (displace? right))
        (with-meta* left
                    (c/merge (-> right meta* (dissoc :displace))
                             (-> left meta* (dissoc :replace))))))

(defrecord Accumulation [value])

;;
;; public api
;;

(defn accumulate [left right _]
  (if (instance? Accumulation left)
    (update left :value conj right)
    (->Accumulation [left right])))

(defn merge
  ([] {})
  ([left] left)
  ([left right] (merge left right nil))
  ([left right {:keys [::path ::paths] :as options}]
   (let [path-f (get paths path)]
     (cond
       path-f
       (path-f left right options)

       (different-priority? left right)
       (pick-prioritized left right options)

       (and (map? left) (map? right))
       (let [merge-entry (fn [m e]
                           (let [k (key e) v (val e)]
                             (if (contains? m k)
                               (assoc m k (merge (get m k) v (update options ::path (fnil conj []) k)))
                               (assoc m k v))))
             merge2 (fn [m1 m2]
                      (reduce merge-entry (or m1 {}) (seq m2)))]
         (reduce merge2 [left right]))

       (and (set? left) (set? right))
       (set/union right left)

       (and (coll? left) (coll? right))
       (if (or (-> left meta :prepend)
               (-> right meta :prepend))
         (-> (into (empty left) (concat right left))
             (with-meta (c/merge (meta left)
                                 (select-keys (meta right) [:displace]))))
         (into (empty left) (concat left right)))

       :else right))))

;;
;; spike
;;

(comment
 (merge
  {:parameters {:query {:x 1}}}
  {:parameters {:query nil}}
  {::replace-nil true})
 ; => {:parameters {:query nil}}
 )

(ns demo)

(require '[ctrl.merge :as cm])

(cm/merge
 {:parameters {:query {:x 1}}
  :get {:parameters {:query {:x 2}}}}
 {:parameters {:query {:y 3}}
  :get {:parameters {:query {:y 4}}}
  :post {:parameters {:query {:y 5}}}}
 {::cm/paths {[:parameters :query] cm/accumulate
              [:get :parameters :query] cm/accumulate
              [:post :parameters :query] cm/accumulate}})
;{:parameters {:query #ctrl.merge.Accumulation{:value [{:x 1} {:y 3}]}},
; :get {:parameters {:query #ctrl.merge.Accumulation{:value [{:x 2} {:y 4}]}}},
; :post {:parameters {:query {:y 5}}}}


(cm/merge
 {:parameters {:query [:map [:x :int]]}
  :get {:parameters {:query [:map [:x :int]]}}}
 {:parameters {:query [:map [:y :int]]}
  :get {:parameters {:query [:map [:y :int]]}}
  :post {:parameters {:query [:map [:y :int]]}}}
 {::cm/paths {[:parameters :query] cm/accumulate
              [:get :parameters :query] cm/accumulate
              [:post :parameters :query] cm/accumulate}})
;{:parameters {:query #ctrl.merge.Accumulation{:value [[:map [:x :int]] [:map [:y :int]]]}},
; :get {:parameters {:query #ctrl.merge.Accumulation{:value [[:map [:x :int]] [:map [:y :int]]]}}},
; :post {:parameters {:query [:map [:y :int]]}}}
