(ns palikka.components.database
  (:require [com.stuartsierra.component :as component]
            [suspendable.core :as suspendable]
            [hikari-cp.core :as hikari]
            [schema.core :as s]
            [schema-tools.core :as st]
            [clojure.tools.logging :as log]
            [palikka.coerce :as c]))

(s/defschema Config
  (assoc (st/optional-keys hikari/ConfigurationOptions)
         :port-number s/Num
         s/Keyword s/Str))

(defrecord Database [config db]
  component/Lifecycle
  (start [this]
    (let [{:keys [server-name port-number] :as c} config]
      (log/infof "Connecting to database on %s:%s" server-name port-number)
      (if-not db
        (assoc this :db {:datasource (hikari/make-datasource c)})
        this)))
  (stop [this]
    (when-let [ds (:datasource db)]
      (try
        (hikari/close-datasource ds)
        (catch Throwable t
          (log/warn t "Error when closing JDBC connection"))))
    (assoc this :db nil))

  suspendable/Suspendable
  (suspend [this]
    this)
  (resume [this old-component]
    (if (= config (:config old-component))
      (assoc this :db (:db old-component))
      (do (component/stop old-component)
          (component/start this)))))

(defn create [config]
  (map->Database {:config (c/env-coerce Config config)}))
