(ns btreekeys.v2
  (:require [clojure.core.async :as a])
  (:import java.util.ArrayList))

(defprotocol QuerySource
  (evaluate-source
    [this query-context]))

(defn collection-source
  [collection]
  (reify QuerySource
    (evaluate-source [this query-context]
      (let [results (a/chan)]
        (a/go
          (doseq [item collection]
            (a/>! results item))
          (a/close! results))
        [query-context results]))))

(defn join-source
  [query join-fn]
  (reify QuerySource
    (evaluate-source [this query-context]
      (let [results (a/chan)
            [query-context query-source] (evaluate-source query query-context)]
        (a/go-loop [query-context query-context]
          (if-let [item (a/<! query-source)]
            (let [new-query-source (join-fn item)
                  [new-query-context join-items] (evaluate-source
                                                   new-query-source
                                                   query-context)]
              (loop [] (when-let [item (a/<! join-items)]
                         (a/>! results item)
                         (recur)))
              (recur new-query-context))
            (a/close! results)))
        [query-context results]))))

(defn transduce-source
  [query xform]
  (reify QuerySource
    (evaluate-source [this query-context]
      (let [results (a/chan)
            [query-context query-source] (evaluate-source query query-context)
            buffer (ArrayList.)
            reduce-f (xform (completing (fn [_ val] (.add buffer val))))]
        (a/go-loop []
          (if-let [item (a/<! query-source)]
            (do (reduce-f nil item)
                (doseq [item buffer]
                  (a/>! results item))
                (.clear buffer)
                (recur))
            (a/close! results)))
        [query-context results]))))

(defn perform-query
  [query]
  (let [[new-context items] (evaluate-source query {})
        buffer (ArrayList.)]
    (loop []
      (when-let [item (a/<!! items)]
        (.add buffer item)
        (recur)))
    [new-context buffer]))

(perform-query
  (-> (collection-source [1 2 3])
      (join-source
        (fn [i] (collection-source (range i))))
      (transduce-source
        (comp (map inc) (filter odd?)))
      (join-source
        (fn [i] (collection-source [i])))))
