(ns mecha1.seed
  (:require
    [com.rpl.specter :as specter]
    [mecha1.seed.impl :as impl]
    [garden.compiler :refer [render-css]]
    [garden.core :as garden]
    [garden.selectors :as gsel :refer [css-selector]]
    [garden.units :as u]
    #?(:cljs [mecha1.seed.inject]))
  #?(:cljs (:require-macros [mecha1.seed])))

(defn css-escape
  "Escapes various characters so that the resulting string is a valid CSS name."
  [x]
  (let [s (if (string? x)
            x
            (str x))]
    (clojure.string/escape s {\: "\\:"
                              \. "\\."
                              \/ "\\/"})))

(defn localize-css-rules
  [rules ns]
  (->> rules
       (specter/transform [impl/TreeValues #(and (not (string? %))
                                            (satisfies? gsel/ICSSSelector %))]
                          #(if-let [[pre n] (impl/class-or-id-selector-components (css-selector %))]
                             (str pre (css-escape (str ns "/" n)))
                             %))))

(defn gen-css
  [& rules]
  (-> (apply garden/css (localize-css-rules rules impl/*named-scope*))
      (clojure.string/replace #"\n\s+\n" "\n\n")))          ; hack to normalize clj/cljs css rendering

#?(:clj
   (defmacro style
     "Defines a set of CSS rules as in noprompt/garden, where any CSS class name or ID keywords
     (e.g. :.foo, :#id) will be replaced with localized qualified names.

     Returns a CSS name resolution function
     which when given an unqualified keyword of a CSS class name or ID used within the ruleset,
     will return the associated qualified string name.

     When the name resolution function is invoked in ClojureScript,
     an inline style element will automatically be injected into the document's head.

     When the name resolution function is invoked in Clojure within the collect-meta macro,
     the inline styles will be appended to the metadata map under the :inline-styles key."
     [& rules]
     (let [s (str *ns* ":" (.hashCode rules))]
       (if (impl/cljs-env? &env)
         ; cljs
         `(let [css# (binding [impl/*named-scope* ~s]
                       (gen-css ~@rules))
                class-names# (impl/map-class-names ~s ~@rules)]
            (letfn [(fn#
                      ([]
                       (mecha1.seed.inject/inject-inline-style! ~s css#))
                      ([& xs#]
                       (fn#)
                       (let [m# (select-keys class-names# xs#)
                             unmatched-keys# (remove #(get m# %) xs#)]
                         (if-not (empty? unmatched-keys#)
                           (js/console.warn ~s "styles do not contain" (pr-str unmatched-keys#))
                           (clojure.string/join " " (vals m#))))))]
              fn#))
         ; clj
         `(let [css# (binding [impl/*named-scope* ~s]
                       (gen-css ~@rules))
                class-names# (impl/map-class-names ~s ~@rules)]
            (letfn [(fn#
                      ([]
                       (when impl/*seed-packet*
                         (swap! impl/*seed-packet* assoc-in [:inline-styles ~s] css#)))
                      ([& xs#]
                       (fn#)
                       (clojure.string/join " " (vals (select-keys class-names# xs#)))))]
              fn#))))))

#?(:clj
   (defmacro use-css
     "Use one or more CSS files.

     When this is called from ClojureScript,
     it will automatically inject a link element referencing the CSS files in the document's head.

     When this is called from Clojure from within the collect-meta macro,
     it will conj the CSS file paths to a set under the :css-files key."
     [& xs]
     (if (impl/cljs-env? &env)
       ; cljs
       `(doseq [x# [~@xs]]
          (mecha1.seed.inject/inject-css-link! x#))
       ; clj
       `(when impl/*seed-packet*
          (swap! impl/*seed-packet*
                 update-in
                 [:css-files]
                 #(apply conj (or %1 []) %2)
                 [~@xs])))))

#?(:clj
   (defmacro use-js
     "Use one or more JS files.

     When this is called from ClojureScript,
     it will automatically inject a link element referencing the CSS files in the document's head.

     When this is called from Clojure from within the collect-meta macro,
     it will conj the CSS file paths to a set under the :css-files key."
     [& xs]
     (if (impl/cljs-env? &env)
       ; cljs
       `(doseq [x# [~@xs]]
          (mecha1.seed.inject/inject-js-script! x#))
       ; clj
       `(when impl/*seed-packet*
          (swap! impl/*seed-packet*
                 update-in
                 [:js-files]
                 #(apply conj (or %1 []) %2)
                 [~@xs])))))

(defn css-join
  "Renders each element in the provided collection with garden.compiler/render-css
  and joins them together into a single string with the rendered values separated with the given separator."
  [sep coll]
  (clojure.string/join sep (map render-css (keep identity coll))))

(defn render-css-units
  "Takes a CSS style map and converts any CSSUnit values within it to strings rendered with garden.compiler/render-css.
  This is a utility function to allow CSSUnits to be used in frameworks like Reagent that do not directly support
  CSSUnits within their CSS style maps."
  [m]
  (specter/transform [specter/MAP-VALS]
                     #(if (u/unit? %)
                        (render-css %)
                        %)
                     m))

(defn root-elem-attrs
  "Takes a component attribute map and returns a map with just the :class and :style key/values from it
  so they can be applied to the component's root element.
  Can optionally be passed a custom attribute map for the root element as the second argument,
  in which case this will be merged with the :style and :class attributes from the component attribute map.

  Attributes from the component attribute map will override those from the root element attribute map.
  Any CSSUnits in the :styles map will be rendered to strings as part of this call."
  ([ca] (root-elem-attrs ca nil))
  ([ca ra]
   (let [{:keys [class style]} (select-keys ca [:class :style])
         m (cond-> ra
                   class (update :class #(css-join " " [% class]))
                   style (update :style merge style))]
     (if (:style m)
       (update m :style render-css-units)
       m))))

(defn css-class-ref
  "Given a CSS class name string (or keyword), returns a properly escaped string reference to that class name."
  [css-class]
  (str "." (css-escape css-class)))

(defn css-id-ref
  "Given a CSS ID string (or keyword), returns a properly escaped string reference to that ID."
  [css-id]
  (str "#" (css-escape css-id)))
