(ns mecha1.seed.inject
  (:require [mecha1.seed.impl :as impl]))

(defn update-inner-html!
  "Updates the given element with the given content."
  [element content]
  (when-not (= (.-innerHTML element) content)
    (set! (.-innerHTML element) content)))

(defn set-attrs-from-map
  [element m]
  (doseq [[k v] m]
    (.setAttribute element (name k) v)))

; Inline style

(defn create-inline-style-element!
  "Creates a new inline style element with the given style content in the document's head."
  [id style]
  (let [head (.-head js/document)
        element (doto (.createElement js/document "style")
                  (.setAttribute "id" id))]
    (assert (some? head)
            "An head element is required in the dom to inject the style.")
    (.appendChild head element)
    (update-inner-html! element style)))

(defn inject-inline-style!
  [id style]
  (if-let [style-element (.getElementById js/document id)]
    (update-inner-html! style-element style)
    (create-inline-style-element! id style)))

; CSS link

(defn create-css-link-element!
  "Creates a new CSS link element with the given style content in the document's head."
  [m]
  (let [head (.-head js/document)
        element (doto (.createElement js/document "link")
                  (set-attrs-from-map m))]
    (assert (some? head)
            "An head element is required in the dom to inject the CSS link.")
    (.appendChild head element)))

(defn inject-css-link!
  [map-or-href]
  (let [{:keys [href] :as m} (if (map? map-or-href)
                               map-or-href
                               {:href map-or-href})
        id (impl/css-link-id href)]
    (when-not (.getElementById js/document id)
      (create-css-link-element! (assoc m
                                  :id id
                                  :rel "stylesheet"
                                  :type "text/css")))))

; JS script

(defn create-js-script-element!
  "Creates a new JS script element with the given style content in the document's body."
  [m]
  (let [body (.-body js/document)
        element (doto (.createElement js/document "script")
                  (set-attrs-from-map m))]
    (assert (some? body)
            "A body element is required in the dom to inject the JS script tag.")
    (.appendChild body element)))

(defn inject-js-script!
  [map-or-href]
  (let [{:keys [src] :as m} (if (map? map-or-href)
                              map-or-href
                              {:src map-or-href})
        id (impl/js-script-id src)]
    (when-not (.getElementById js/document id)
      (create-js-script-element! (assoc m
                                   :id id
                                   :type "text/javascript")))))
