(ns mecha1.seed.impl
  (:require
    [com.rpl.specter :as specter]
    [garden.core :as garden]
    [garden.selectors :as gsel :refer [css-selector]]))

(defn at-rule?
  "Returns true if the argument is a garden.types.CSSAtRule, false otherwise."
  [x]
  (instance? garden.types.CSSAtRule x))

(def TreeValues
  (specter/recursive-path [] p
                          (specter/cond-path
                            sequential? [specter/ALL p]
                            at-rule? [:value :rules p]
                            :else specter/STAY)))

(defn class-or-id-selector-components
  "Get the prefix and name of the class or ID referenced in the given selector string, e.g.:
    \"#foo\" => [\"#\" \"foo\"]
    \".thing\" => [\".\" \"thing\"]
    \"&.xyz\" => [\"&.\" \"xyz\"]

  Returns nil if the selector is not a valid class or ID selector."
  [s]
  (let [c (first s)]
    (cond
      (#{\# \.} c) [(str c) (subs s 1)]
      (and (= c \&)
           (= (second s) \.)) ["&." (subs s 2)])))

(def ^:dynamic *named-scope* nil)

(defn map-class-names
  "Given a namespace string and a set of rules,
  finds any unqualified CSS selector keywords in the ruleset
  and returns a map from keywords to namespace-qualified non-escaped strings
  for each such CSS selector found."
  [ns & rules]
  (into {}
        (map (fn [sel]
               (when-let [[_ n] (class-or-id-selector-components (css-selector sel))]
                 [(keyword n)
                  (str ns "/" n)])))
        (specter/select [TreeValues #(and (not (string? %))
                                          (satisfies? gsel/ICSSSelector %))]
                        rules)))

(defn cljs-env?
  "Take the &env from a macro, and tell whether we are expanding into cljs."
  [env]
  (boolean (:ns env)))

#?(:clj (def ^:dynamic *seed-packet* nil))

(defn css-link-id [href] (str :mecha1.seed/css ":" href))

(defn js-script-id [src] (str :mecha1.seed/js ":" src))
