(ns mecha1.seed.gen-static
  (:require
    [mecha1.seed.impl :as impl]))

(defn include-css
  [& xs]
  (for [map-or-href xs]
    (let [{:keys [href] :as m} (if (map? map-or-href)
                                 map-or-href
                                 {:href map-or-href})]
      [:link (assoc m
               :id (impl/css-link-id href)
               :rel  "stylesheet"
               :type "text/css")])))

(defn include-js
  [& xs]
  (for [map-or-href xs]
    (let [{:keys [src] :as m} (if (map? map-or-href)
                                map-or-href
                                {:src  map-or-href})]
      [:script (assoc m
                 :id (impl/js-script-id src)
                 :type "text/javascript")])))

(defmacro collect
  "Wraps forms that generate HTML content and tracks what CSS styles, CSS files, and JS files are referenced during HTML generation.
  Returns a map with :inline-styles, :css-files, :js-files, and :content keys.
  The value of the :inline-styles key is a map of inline style element IDs to inline style contents.
  The value of the :css-files key is a set of CSS file paths.
  The value of the :js-files key is a set of JS file paths.
  The value of the :content key is the result of the evaluating the wrapped forms in an implicit do."
  [& contents]
  `(binding [impl/*seed-packet* (atom {})]
     (let [contents# (do ~@contents)]
       (assoc @impl/*seed-packet* :contents contents#))))

(defn apply-seeds
  "Applies collected info from mecha1.seed.gen-static/collect to the given head and body elements.

  The first argument should be a map returned by mecha1.seed.gen-static/collect.
  The second and third arguments should be hiccup data structures for the [:head] element and [:body] element.

  This function will render and append :css-files and :inline-styles to the head element,
  and :contents and :js-files to the body element.

  Returns a list whose first element is the modified [:head] and second element is the modified [:body]."
  [{:keys [inline-styles
           css-files
           contents
           js-files]}
   head
   body]
  (list (conj head
              (for [css-file css-files]
                (include-css css-file))
              (for [[id style] inline-styles]
                [:style {:id id} style]))
        (conj body
              contents
              (for [js-file js-files]
                (include-js js-file)))))
