(ns mecha1.cljs-modules
  (:require
    [cljs.core.async :as async :refer [<!]]
    [goog.module.ModuleManager]
    [goog.module.ModuleLoader]
    [pushy.core]
    [taoensso.timbre :as log :include-macros true])
  (:require-macros
    [cljs.core.async.macros :refer [go]]))

(goog-define dev-mode? false)

(def manager (atom nil))

(defn init-module-manager
  "Initializes the module manager.
  Takes a map of module names to module dependencies,
  and a second map of module names to module URIs."
  [module-deps module-uris]
  (reset! manager (doto (goog.module.ModuleManager/getInstance)
                    (.setLoader (goog.module.ModuleLoader.))
                    (.setAllModuleInfo (clj->js module-deps))
                    (.setModuleUris (clj->js module-uris)))))

(defn require-module
  "Loads module from the network if necessary. Always returns a
  channel that will be closed when the module is loaded (sometimes immediately)"
  [id]
  (let [chan (async/chan)]
    (if dev-mode?
      (async/close! chan)
      (.execOnLoad @manager id #(async/close! chan)))
    chan))

(defn set-loaded!
  "Inform the module manager the the named module is loaded."
  [module-name]
  (go
    (while (nil? @manager)
      (<! (async/timeout 100)))
    (.setLoaded @manager module-name)))

; ----------------------------------------------------------------------------------------

(defmulti page-for-route
          "Returns the page associated with the given route"
          first)                                            ; <- oops. this refers to the structure of a bide route.

(defn init
  [{:keys [::active-page-atom ::match-fn
           :pushy/processable-url? :pushy/identity-fn
           ::done-fn]}]
  (let [dispatch-fn (fn dispatch [{:keys [module route]}]
                      (log/debug "Dispatching to route" route "in module" module)
                      (go
                        (<! (require-module module))
                        (reset! active-page-atom ((page-for-route route)))))
        history (apply pushy.core/pushy
                       dispatch-fn
                       match-fn
                       (cond-> []
                               processable-url? (conj :processable-url? processable-url?)
                               identity-fn (conj :identity-fn identity-fn)))]
    (pushy.core/start! history)
    (add-watch active-page-atom :init
               (fn [key ref old-state new-state]
                 (when (and (= key :init) (= old-state nil))
                   (done-fn))))))
