(ns mecha1.boot-cljs-modules
  (:require
    [boot.core :refer [deftask]]
    [boot.util]
    [com.rpl.specter :refer :all]))

(defn ns-str->path
  "Takes the name of a Clojure namespace and returns the associated filename path, minus the file extension."
  [p]
  (-> p
      (clojure.string/replace "." "/")
      (clojure.string/replace "-" "_")))

(defn find-file
  "Searches the fileset for TmpFiles with the given path plus any of the given extensions. Returns the first match."
  [fs path-minus-ext exts]
  (some (fn [ext]
          (boot.core/tmp-get fs (str path-minus-ext "." ext)))
        exts))

(defn create-file
  "Creates a file with the given base dir and path, creating any parent directiories if they do not already exist."
  [base-dir path]
  (doto (clojure.java.io/file base-dir path)
    (-> .getParentFile .mkdirs)))

(defn rewrite-cljs-edn
  "Modifies the .cljs.edn file by adding each module's namespace to its list of entries.

  If dev-mode? is true, will add all module namespaces to :require so that they will be immediately loaded when the js file is loaded.
  This is useful for development when modules are not present."
  [resource-dir path cljs-edn dev-mode?]
  (spit (create-file resource-dir path)
        (pr-str (->> (if dev-mode?
                       (->> (assoc-in cljs-edn [:compiler-options :closure-defines 'mecha1.cljs-modules/dev-mode?] true)
                            ; add all module namespaces to :require
                            (transform [(collect :compiler-options :modules MAP-KEYS (view (comp symbol name)))
                                        :require]
                                       #(apply conj %2 %1)))
                       cljs-edn)
                     ; add module namespace to entries
                     (transform [:compiler-options :modules ALL (collect-one FIRST (view name)) LAST :entries]
                                #(conj %2 %1))))))

(defn create-modules-namespace
  "Creates code for a modules namespace that initializes the module manager with module dependency information and paths to module js files,
  and contains configuration for module routes.

  The name of the modules namespace is taken from the name of the .cljs.edn file.
  For example, if the file is named 'example.app.cljs.edn', the associated module namespace will be named 'example.app.cljs-modules'.

  The module routes are discovered by looking in the fileset for the TmpFiles for each module's entries and checking for :mecha1.web generator directives attached to them."
  [fileset tmpfile source-dir cljs-edn]
  (let [ns-str (-> (boot.core/tmp-file tmpfile)
                   .getName
                   (clojure.string/replace #"\.cljs\.edn$" ".cljs-modules"))]
    (doto (clojure.java.io/file source-dir (str (ns-str->path ns-str) ".cljs"))
      (-> .getParentFile .mkdirs)
      (spit (str (pr-str (list 'ns (symbol ns-str)
                               `(:require mecha1.cljs-modules))) "\n"
                 ; init module manager
                 (pr-str `(mecha1.cljs-modules/init-module-manager
                            ; map of module names to module dependencies
                            ~(->> cljs-edn
                                  (select [:compiler-options :modules ALL (collect-one FIRST (view name)) LAST :depends-on NIL->VECTOR])
                                  (into {}))
                            ; map of module names to module js uris
                            ~(->> cljs-edn
                                  (select [:compiler-options :modules ALL FIRST (view name)])
                                  (map (fn [ns-str]
                                         [ns-str (format "/%s/%s.js"
                                                         (clojure.string/replace (:path tmpfile) #"\.cljs\.edn$" "")
                                                         ns-str)]))
                                  (into {})))) "\n"
                 ; module routes
                 (pr-str (list 'def 'module-routes
                               ; map of module to bide routes
                               (->> cljs-edn
                                    (select [:compiler-options :modules ALL (collect-one FIRST (view name))
                                             LAST :entries
                                             (transformed ALL
                                                          #(get (find-file fileset (ns-str->path %) ["clj" "cljc"])
                                                                :mecha1.web/url))
                                             (filterer identity)])
                                    (into {})))))))))

(defn create-individual-module-namespaces
  "Creates a module namespace for each module in the .cljs.edn configuration.

  The name of the module namespace is taken from the name of the module keyword in the .cljs.edn configuration.
  For example, the following .cljs.edn configuration will create namespaces 'example.module.inner' and 'example.module.outer':

  {:modules {:example.module.inner {:entries [...]}
             :example.module.outer {:entries [...]}}}

  Each generated module namespace requires all of the namespace entries in the module
  and tells the module loader that that the module has been loaded."
  [source-dir cljs-edn]
  (doseq [[ns-str requires] (select [:compiler-options :modules ALL (collect-one FIRST (view name)) LAST :entries (transformed ALL symbol)]
                                    cljs-edn)]
    (spit (create-file source-dir (str (ns-str->path ns-str) ".cljs"))
          (str (pr-str (list 'ns (symbol ns-str)
                             `(:require mecha1.cljs-modules ~@requires))) "\n"
               (pr-str `(mecha1.cljs-modules/set-loaded! ~ns-str))))))

(deftask cljs-modules
         "Support for ClojureScript modules; intended to be used in conjunction with mecha1.cljs-modules.

         Scans the fileset for .cljs.edn files and uses the :modules configuration within :compiler-options to generate module support code.
         This task adds the following additional semantics to .cljs.edn files:

         The name of the .cljs.edn file is used to generate the name of a modules namespace which is used to initialize and configure module functionality.
         For example, if the file is named 'example.app.cljs.edn', the associated module namespace will be named 'example.app.cljs-modules'.

         The module identifier keywords in the :modules configuration are used to name individual module namespaces.
         For example, the following .cljs.edn configuration will create namespaces 'example.module.inner' and 'example.module.outer':

         {:compiler-options {:modules {:example.module.inner {:entries [...]}
                                       :example.module.outer {:entries [...]}}}

         Module :entries should be contained in an ordered list (i.e. vector) in order for their associated routes to be generated in order.
         Routes for the application are discovered by looking in the fileset for the TmpFiles for each module's entries
         and checking for :mecha1.web generator directives attached to them.

         NOTE: The mecha1.boot-file-meta/file-meta task should be composed ahead of this task
         and namespaces should be annotated with :mecha1.web generator directives
         for route discovery to function appropriately."
         [d dev-mode bool "Development mode: modules are disabled. Defaults to false."
          o output bool "Outputs generated source files which are not normally output. Useful for debugging."]
         (let [source-dir (boot.core/tmp-dir!)
               resource-dir (boot.core/tmp-dir!)]
           (fn [next-handler]
             (fn [fileset]
               (boot.core/empty-dir! source-dir)
               (boot.core/empty-dir! resource-dir)
               (doseq [{:keys [path role] :as tmpfile} (->> fileset
                                                            boot.core/input-files
                                                            (boot.core/by-ext #{".cljs.edn"}))]
                 (let [cljs-edn (->> tmpfile
                                     boot.core/tmp-file
                                     slurp
                                     clojure.edn/read-string)]
                   (rewrite-cljs-edn resource-dir path cljs-edn dev-mode)
                   (create-modules-namespace fileset tmpfile (if output resource-dir source-dir) cljs-edn)
                   (create-individual-module-namespaces (if output resource-dir source-dir) cljs-edn)))
               (-> fileset
                   (boot.core/add-source source-dir)
                   (boot.core/add-resource resource-dir)
                   boot.core/commit!
                   next-handler)))))

(defn path->ns-str
  "Takes a .clj or .cljc filename path and returns the name of the associated Clojure namespace."
  [p]
  (-> p
      (clojure.string/replace #"\.cljc?$" "")
      (clojure.string/replace "/" ".")
      (clojure.string/replace "_" "-")))

(defn append-page-for-route-defmethod
  [{:keys [mecha1.web/url
           mecha1.web/cljs-page
           mecha1.web/page
           path]
    :as   tmpfile}
   source-dir]
  (let [page-component (or cljs-page page)
        page-component-ns (or (some-> page-component
                                      namespace)
                              (path->ns-str (boot.core/tmp-path tmpfile)))
        page-component-name (or (some-> page-component
                                        name)
                                "page")
        page-component-sym (symbol page-component-ns page-component-name)
        f (clojure.java.io/file source-dir path)]
    (.mkdirs (.getParentFile f))
    (clojure.java.io/copy (boot.core/tmp-file tmpfile) f)
    (spit f
          (str "\n#?(:cljs "
               (pr-str (list 'defmethod 'mecha1.cljs-modules/page-for-route url '[_] page-component-sym))
               ")")
          :append true)))

(deftask gen-page-for-route-methods
         "Searches the fileset for any .cljc files with :mecha1.web/url generator directive metadata,
         and appends a generated mecha1.cljs-modules/page-for-route defmethod to each.

         The generated page-for-route defmethod will use the mecha1.web/url for its dispatch value,
         and will return the mecha1.web/page symbol value if it exists, or a function named 'page' if it does not.
         If the page component symbol is not namespace qualified, it will be interpreted as belonging in the namespace of the file."
         [o output bool "Outputs generated source files which are not normally output. Useful for debugging."]
         (let [output-dir (boot.core/tmp-dir!)]
           (fn [next-handler]
             (fn [fileset]
               (boot.core/empty-dir! output-dir)
               (doseq [tmpfile (->> fileset
                                    boot.core/input-files
                                    (boot.core/by-ext #{".cljc"})
                                    (boot.core/by-meta #{:mecha1.web/url}))]
                 (append-page-for-route-defmethod tmpfile output-dir))
               (let [add-output-dir (if output
                                      boot.core/add-resource
                                      boot.core/add-source)]
                 (-> fileset
                     (add-output-dir output-dir)
                     boot.core/commit!
                     next-handler))))))
