(ns mecha1.boot-gen-html
  (:require
    [boot.core :refer [deftask]]
    [boot.util]
    [clojure.string]))

(defn path->ns-str
  "Takes a .clj or .cljc filename path and returns the name of the associated Clojure namespace."
  [p]
  (-> p
      (clojure.string/replace #"\.cljc?$" "")
      (clojure.string/replace "/" ".")
      (clojure.string/replace "_" "-")))

(defn require-and-resolve
  "Requires and resolves the given symbol and returns the result.
  Qualifies the symbol with the default namespace if it is not already namespace qualified.
  If the symbol is nil, will resolve the default symbol name in the default namespace."
  ([sym] (require-and-resolve sym nil nil))
  ([sym default-ns default-name]
   (let [sym-ns (or (some-> sym
                            namespace)
                    default-ns)]
     (try
       (require (symbol sym-ns))
       (catch Throwable e
         (boot.util/exit-error
           (boot.util/fail "Unable to require namespace %s: %s\n"
                           sym-ns
                           (.getMessage e)))))

     (let [sym-name (or (some-> sym
                                name)
                        default-name)
           qualified-sym (symbol sym-ns sym-name)]
       (or (resolve qualified-sym)
           (boot.util/exit-error
             (boot.util/fail "Unable to resolve %s\n" qualified-sym)))))))

(defn generate-html
  [{:keys [mecha1.web/url
           mecha1.web/clj-page
           mecha1.web/page]
    :as   tmpfile}
   outdir
   generator-fn]
  (when-let [component-var (require-and-resolve (or clj-page page)
                                                (path->ns-str (boot.core/tmp-path tmpfile))
                                                "page")]
    (let [f (clojure.java.io/file outdir (subs url 1) "index.html")
          filename (subs (.getPath f) (count (.getPath outdir)))]
      (boot.util/info "Generating %s\n" filename)
      (.mkdirs (.getParentFile f))
      (spit f (generator-fn (var-get component-var))))))

(deftask gen-html
         "Searches the fileset for any input files with mecha1.web generator directive metadata,
         and generates html files by invoking the specified HTML generator function on them.

         Supported generator directives are:

         :mecha1.web/url [required] - String URL path for this page. Should start with a '/' character.

         :mecha1.web/clj-page [optional] - A symbol identifying the page component for this file.
           Will be provided as a parameter to the HTML generator function.
           If the symbol is not namespace qualified, the namespace of the symbol will be assumed to be the namespace of the current file.
           If no page component is specified, defaults to a function named 'page' within the file's namespace.

         :mecha1.web/page [optional] - Same as :mecha1.web/clj-page.
           To be used when the same function is used to render server-side content in Clojure
           and client-side content in ClojureScript.
           Ignored by server-side HTML generator if :mecha1.web/clj-page is also set."
         [g generator FN sym "HTML generator function. Takes the page component and returns the HTML file contents."]
         (if-not generator
           (boot.util/exit-error (boot.util/fail "HTML generator function not specified"))
           (let [asset-dir (boot.core/tmp-dir!)]
             (fn [next-handler]
               (fn [fileset]
                 (boot.core/empty-dir! asset-dir)
                 (doseq [{:keys [mecha1.web/url]
                          :as   tmpfile} (->> fileset
                                              boot.core/input-files
                                              (boot.core/by-meta #{:mecha1.web/url}))]
                   (if (clojure.string/starts-with? url "/")
                     (if-let [generator-fn (require-and-resolve generator)]
                       (generate-html tmpfile asset-dir generator-fn)
                       (boot.util/exit-error (boot.util/fail "Unable to resolve generator function: %s\n" generator)))
                     (boot.util/exit-error
                       (boot.util/fail "In file %s, :mecha1.web/url metadata value %s should start with a '/' character.\n"
                                       url
                                       (boot.core/tmp-path tmpfile)))))
                 (-> fileset
                     (boot.core/add-asset asset-dir)
                     boot.core/commit!
                     next-handler))))))
