(ns mecha1.boot-file-meta
  (:import
    [java.io PushbackReader]
    [java.util.regex Pattern])
  (:require
    [boot.core :refer [deftask]]))

(defn read-file-metadata
  "Returns the metadata map from the first lines of the given file,
  or nil if the file does not start with any such metadata map.
  The provided prefix should be a java.util.Pattern regex string; special characters should be escaped accordingly."
  [f prefix]
  (with-open [reader (clojure.java.io/reader f)]
    (.mark reader 1000)
    (when-let [[_ skip] (re-matches (Pattern/compile (str "(\\s*#_\\s*)" prefix ".*"))
                                    (.readLine reader))]
      (.reset reader)
      (.skip reader (count skip))                           ; skip the ignore prefix #_
      (clojure.edn/read (PushbackReader. reader)))))

(deftask file-meta
         "Searches the fileset for files with embedded file metadata in the form of an ignored map form prefixed with #_
         starting on the first line of the file.
         If such a metadata map form is found, it is merged into the metatadata for that file in the fileset.
         Note that the prefix #_ indicates that the map will be discarded by the Clojure reader,
         so it will not affect the rest of the code in the file.

         Searched files may be restricted by extension, or by a regex matching the first line of the file after the '#_' characters.
         Note that the regex only matches the first line of the file, but the form itself may span multiple lines, e.g.:

         #_#:mecha1.web
           {:url \"/\"
            :page page}

         Here the metadata is a namespaced map form where the keys in the map belong to the mecha1.web namespace."
         [e exts EXT #{str} "The set of file extensions that this task will limit itself to."
          r regex REGEX str "The regex to match the first line of the file, immediately after the '#_' characters. Only files that match will be processed."]
         (let [regex (or regex "")
               tmp (boot.core/tmp-dir!)]
           (fn [next-handler]
             (fn [fileset]
               (boot.core/empty-dir! tmp)
               (let [meta-map (into {}
                                    (map (fn [tmp-file]
                                           (when-let [file-meta (-> (boot.core/tmp-file tmp-file)
                                                                    (read-file-metadata regex))]
                                             (when (map? file-meta)
                                               [(:path tmp-file) file-meta]))))
                                    (cond->> (boot.core/input-files fileset)
                                             (not-empty exts) (boot.core/by-ext exts)))]
                 (-> fileset
                     (boot.core/add-meta meta-map)
                     next-handler))))))
