(ns ring.middleware.mime-versioned
  "Middleware that provides versioning information to a handler
  by using the 'Accept' HTTP header and Vendor MIME Types."
  (:require [ring.util.response :refer [get-header]]))

;; Define a regular expression matching a Vendor MIME Type.
;; Example matching string: "application/vnd.colindrake-v1+json"
;; Source: http://tools.ietf.org/html/rfc4288#section-3.2
(def versioned-regex #"(.+)/vnd.(.+)-(v.+)\+(.+)")

(defn- extract-version-components
  "Returns the version components of a vendor Accept header
  if found. Otherwise, returns nil.

  s is a string and this function returns a map."
  [s]
  (when s
    (when-let [components (re-find versioned-regex s)]
      (let [[_ mime-type vendor version mime-subtype] components]
        {:bare-mime-type (str mime-type "/" mime-subtype) :mime-vendor vendor :version version}))))

(defn wrap-mime-versioned
  "Wraps a Ring handler and provides API version information
  in the request map. Assumes a Vendor MIME Type versioning
  schema in the 'Accept' HTTP header. The header may be set to
  a value such as 'application/vnd.<vendor>-v<version>+json'.

  Upon detecting such a header, this middleware layer will merge
  in 'bare-mime-type', 'mime-vendor', and 'version' keys.
  "
  [handler]
  (fn [request]
    (let [accept-value (get-header request "Accept")
          components (extract-version-components accept-value)]
      (if components
        (handler (merge request components))
        (handler request)))))
