(ns me.colindrake.hiro
  (:require [clojure.java.shell :refer [sh]]
            [clojure.data.json :as json]))

;;; Shell Access
;;; Functions and values to aid in executing the snowcrash parser.
(def ^:private initial-command ["drafter" "-f" "json"])

(defn- exec-snowcrash
  "Executes the snowcrash parser returning a JSON representation of the API, or nil if unable to parse."
  [args input-str]
  (let [command (concat initial-command args)
        command-with-input (concat command [:in input-str])
        result (apply sh command-with-input)
        exit-code (:exit result)
        output (:out result)]
    (when (= exit-code 0)
      (json/read-str output))))

;;; Parsing
;;; Functions to parse the output of the snowcrash parser into Clojure data structures.
(defn- parse-blueprint-json
  "Parses a string containing an API Blueprint into a JSON representation."
  [api-blueprint-str]
  (exec-snowcrash [] api-blueprint-str))

(defn- get-title
  "Returns the title element from a category."
  [blueprint-json]
  (get-in blueprint-json ["meta" "title"]))

(defn- get-attributes
  "Returns all attributes in a key-value (map) form."
  [blueprint-json]
  (let [meta (get-in blueprint-json ["attributes" "meta"])
        pair-fn (fn [e] [(get-in e ["content" "key" "content"]) (get-in e ["content" "value" "content"])])]
    (into {} (map pair-fn meta))))

(declare parse-element)

(defn- parse-category
  "Distills a category down to the essentials."
  [element]
  {:type :category
   :title (get-title element)
   :attributes (get-attributes element)
   :items (map parse-element (get element "content"))})

(defn- parse-copy
  "Distills copy down to the essentials."
  [element]
  {:type :copy :content (get element "content")})

(defn- parse-resource
  "Distills a resource down to the essentials."
  [element]
  (assoc (parse-category element) :type :resource))

(defn- parse-element
  "Parses a piece of JSON from the API Blueprint JSON representation."
  [element]
  (let [type (get element "element")]
    (case type
      "category" (parse-category element)
      "copy" (parse-copy element)
      "resource" (parse-resource element)
      {}))) ; TODO: Add more!

(defn parse-blueprint
  "Parses a string containing an API Blueprint into a Clojure-friendly representation."
  [api-blueprint-str]
  (when api-blueprint-str
    (let [json (parse-blueprint-json api-blueprint-str)]
      (parse-element json))))
