(ns clj-infra.digitalocean
  (:require [clj-http.client :as http]
            [cheshire.core :as json]))

(def root "https://api.digitalocean.com/v2/")

(def registry-root "registry.digitalocean.com")

;; HTTP Primitives

(defn- auth [token]
  {:Authorization (format "Bearer %s" token)})

(defn- svc-url [dest]
  (format "%s/%s" root dest))

(defn- http-post [token dest body]
  (let [{:keys [body]}
        (http/post (svc-url dest)
                   {:accept :json
                    :content-type :json
                    :headers (auth token)
                    :form-params body})]
    (json/parse-string body true)))

(defn- http-delete [token dest]
  (http/delete (svc-url dest) {:headers (auth token)}))

(defn- http-put [token dest body]
  (http/put (svc-url dest)
            {:content-type :json
             :headers (auth token)
             :body (json/generate-string body)}))

(defn- http-get [token dest]
  (let [{:keys [body]}
        (http/get (svc-url dest) {:headers (auth token)})]
    (json/parse-string body true)))

;; Core operations

;;; Container Registries

(defn create-container-registry [token region registry-name]
  (let [result
        (http-post token "registry"
                   {:name registry-name
                    :region region
                    :subscription_tier_slug "starter"})]
    (merge result {:url (format "%s/%s" registry-root registry-name)})))

(defn delete-container-registry [token]
  (http-delete token "registry"))

;;; Volumes

(defn create-storage-volume [token region volume-name size-gb]
  (http-post token "volumes"
             {:name volume-name
              :region region
              :size_gigabytes size-gb
              :filesystem_type "ext4"}))

(defn delete-storage-volume [token region volume-name]
  (http-delete token (format "volumes?name=%s&region=%s" volume-name region)))

;;; SSH Keys

(defn upload-ssh-public-key [token key-name key]
  (http-post token "account/keys" {:public_key key, :name key-name}))

(defn remove-ssh-public-key [token ssh-key-id]
  (http-delete token (format "account/keys/%s" ssh-key-id)))

;;; Droplets

(defn create-simple-droplet [token region droplet-name droplet-size droplet-image ssh-key-id tags]
  (http-post token "droplets"
             {:name droplet-name
              :region region
              :size droplet-size
              :image droplet-image
              :ssh_keys [ssh-key-id]
              :backups false
              :monitoring true
              :user_data "#cloud-config\nruncmd:\n - ufw disable\n",
              :tags tags}))

(defn destroy-droplet [token droplet-id]
  ;; One time, this delete succeeded but the droplet was still active.
  ;; @todo Should we consider retrying until the droplet is well and truly
  ;; @todo gone?
  (http-delete token (format "droplets/%s" droplet-id)))

(defn get-droplet-info [token droplet-id]
  (http-get token (str "droplets/" droplet-id)))

(defn get-droplet-ip
  ([token droplet-id]
   (get-droplet-ip (get-droplet-info token droplet-id)))
  ([info]
   (->> info
        :droplet
        :networks
        :v4
        (filter (comp (partial = "public") :type))
        first
        :ip_address)))

(defn get-droplet-status [token droplet-id]
  (-> (get-droplet-info token droplet-id) :droplet :status))

(defn reboot-droplet [token droplet-id]
  (http-post token (format "droplets/%s/actions" droplet-id) {:type "reboot"}))

;; Storage Volume Attachments

(defn attach-storage-to-droplet [token region volume-id droplet-id]
  (http-post token (format "volumes/%s/actions" volume-id)
             {:type "attach"
              :droplet_id droplet-id
              :region region}))

(defn detach-storage-from-droplet [token region volume-id droplet-id]
  (http-post token (format "volumes/%s/actions" volume-id)
             {:type "detach"
              :droplet_id droplet-id
              :region region}))

;;; SSL Certificates

(defn create-ssl-cert [token cert-name dns-names]
  (http-post token "certificates"
             {:name cert-name
              :type "lets_encrypt"
              :dns_names dns-names}))

(defn delete-ssl-cert [token cert-id]
  (http-delete token (format "certificates/%s" cert-id)))

(defn get-ssl-cert-info [token cert-id]
  (http-get token (format "certificates/%s" cert-id)))

;;; Load Balancers

(defn create-load-balancer
  ([token region lb-name ssl-cert-id droplet-id target-port health-check-path]
   (http-post token "load_balancers"
              {:name lb-name
               :region region

               :forwarding_rules
               [{:entry_protocol "https"
                 :entry_port 443
                 :target_protocol "http"
                 :target_port target-port
                 :certificate_id ssl-cert-id}]

               :health_check
               {:protocol "http"
                :port target-port
                :path health-check-path
                :check_interval_seconds 10
                :response_timeout_seconds 5
                :healthy_threshold 5
                :unhealthy_threshold 3}

               :droplet_ids [droplet-id]
               :redirect_http_to_https true
               :disable_lets_encrypt_dns_records true}))
  ([token region lb-name ssl-cert-id droplet-id target-port]
   (create-load-balancer token
                         region
                         lb-name
                         ssl-cert-id
                         droplet-id
                         target-port
                         "/"))

  ([token region lb-name ssl-cert-id droplet-id]
   (create-load-balancer token region lb-name ssl-cert-id droplet-id 80)))

(defn delete-load-balancer [token lb-id]
  (http-delete token (format "load_balancers/%s" lb-id)))

(defn get-load-balancer-info [token lb-id]
  (http-get token (format "load_balancers/%s" lb-id)))

;;; DNS

(defn add-domain-a-records [token ip domain subdomains]
  (let [domains (into ["@"] subdomains)]
    (doall
      (map (fn [d]
             (->> {:type "A"
                   :name d
                   :data ip
                   :priority nil
                   :port nil
                   :ttl 60
                   :weight nil
                   :flags nil
                   :tag nil}
                  (http-post token (format "domains/%s/records" domain))))
           domains))))

(defn delete-domain-records [token domain record-ids]
  (doall
    (map
      (fn [record-id]
        (http-delete token (format "domains/%s/records/%s" domain record-id)))
      record-ids)))

;; Managed Databases

(defn list-database-options [token]
  (http-get token "databases/options"))

(defn create-postgresql-db-cluster
  [token region db-name db-size node-count tags]
  (http-post token "databases"
             {:name db-name
              :engine "pg"
              :region region
              :size db-size
              :num_nodes node-count
              :rules (map (fn [tag] {:type "tag", :value tag}) tags)
              :tags tags}))

(defn delete-postgresql-db-cluster [token dbc-id]
  (http-delete token (format "databases/%s" dbc-id)))

(defn get-dbc-info [token dbc-id]
  (http-get token (format "databases/%s" dbc-id)))

(defn get-dbc-status [token dbc-id]
  (-> (get-dbc-info token dbc-id) :database :status))

(defn add-postgresql-db [token dbc-id database-name]
  (http-post token (format "databases/%s/dbs" dbc-id)
             {:name database-name}))

(defn delete-db [token dbc-id database-name]
  (http-delete token (format "databases/%s/dbs/%s" dbc-id database-name)))