(ns cerber.registrar
  (:require [compojure.core        :refer [routes GET POST OPTIONS ANY]]
            [liberator.core        :refer [resource defresource]]
            [clauth.middleware     :as middleware]
            [clauth.user           :as user]
            [bouncer.core          :as bc]
            [bouncer.validators    :as bv]
            [clojure.tools.logging :as log]
            [cerber.db.store       :as store]
            [cerber.http.response  :as response]
            [cerber.http.resource  :as resource]))

(defn register-user
  "Registers new user based on obligatory fields (login/email) and optional ones (oauth id/pass).
  Initially, newly created user is inactive (unless authenticated by external provider or activated by active? param)
  and has a confirmation token generated. Once confirmed (via link in email), user becomes active."

  [user]
  (user/register-user (store/bare-new-user user)))

(defn validate-params
  "Validates incoming request. If it's GET/OPTIONS - it's perfectly processable.
  Otherwise POST/PUT params are checked against certain validation rules.
  If validations fails HTTP 422 (Unprocessable entity) is returned."

  [{{params :params, method :request-method} :request}]
  (or
   (#{:get :options} method)
   (let [[errors _] (bc/validate params
                                 :login [[bv/required]
                                         [bv/min-count 3]
                                         [bv/max-count 30]
                                         [bv/matches #"^[a-z][a-z0-9\.\_]+$"]
                                         [(complement store/fetch-user-by-login) :message "User already exists"]]
                                 :email [[bv/required]
                                         [bv/max-count 255]
                                         [bv/email]]
                                 :password [[bv/required]
                                            [bv/min-count 5]])]
     [(nil? errors) {:registry/validation-errors errors}])))

(defn validate-existence [{{params :params} :request}]
  (when-let [user (store/fetch-user-by-login (:login params))]
    {::user user}))

(defn user-registration-handler
  "Creates new user via underlaying user-store.
  User may be auto-activated (no email verification required) depending on with-confirmation? flag."

  [config {{params :params} :request}]
  {::user (register-user (assoc params :active? (not (:confirmation-required? config))))})

(defn confirmation-form-handler
  "Renders the confirmation page to acknowledge user's registration"

  [req]
  (response/render-form "auth/confirm.html" {} req))

(defn registration-form-handler
  "Renders registration form which may slightly differ depending on passed auth token.
  Empty token means no external OAuth provider had verified user before, therefore login and password are also left empty."

  [req]
  (response/render-form "auth/register.html" {:url "/login"
                                              :action "/register"
                                              :token ""
                                              :login ""
                                              :email ""} req))

(defn allowed-methods-handler
  "Returns allowed methods on user resource.
  Computation depends on prior authorization and resource existence."

  [{{:keys [params principal]} :request}]
  (let [user (store/fetch-user-by-login (:login params))]
    (response/allowed-methods user principal)))

(defn init-routes
  "Initializes Registrar routes"

  [config]
  (defresource user-resource
    :as-response response/filtered-response
    :available-media-types ["application/json"]
    :allowed-methods [:get :post :options]
    :exists? validate-existence
    :processable? validate-params
    :post! (partial user-registration-handler config)
    :handle-options allowed-methods-handler
    :handle-unprocessable-entity :registry/validation-errors
    :handle-ok ::user
    :handle-created (fn [ctx]
                      (or (::user ctx) (response/user-exists))))

  (defresource principal-resource
    (resource/authorized-resource)
    :as-response response/filtered-response
    :available-media-types ["application/json"]
    :allowed-methods [:get]
    :handle-ok (comp :principal :request))

  (routes
   (ANY     "/confirm"      [] confirmation-form-handler)
   (GET     "/register"     [] (middleware/csrf-protect! registration-form-handler))
   (POST    "/register"     [] (middleware/csrf-protect! user-resource))
   (GET     "/user/status"  [] principal-resource)
   (OPTIONS "/user/:login"  [login] user-resource)
   (GET     "/user/:page"   [page] (partial response/render-page (str "user/" page ".html") {}))))
