(ns cerber.middleware
  (:require [compojure.core                 :refer [wrap-routes]]
            [ring.middleware.defaults       :refer [site-defaults api-defaults wrap-defaults]]
            [ring.middleware.session.cookie :refer [cookie-store]]
            [prone.middleware               :refer [wrap-exceptions]]
            [selmer.middleware              :refer [wrap-error-page]]
            [cerber.config                  :refer [app-config]]
            [clojure.tools.logging          :as log]
            [clauth.token]))

(defn- session-store
  "Defines cookie-based session store.
   secret-key is used as a cookie encryption key."

  [secret-key]
  {:store (cookie-store {:key secret-key})
   :flash false})

(defn- find-token
  "Looks for a authorization token in session or in header"

  [req]
  (when-let [token (or (get-in req [:session :access_token])
                       (second (.split (get-in req [:headers "authorization"] "") " ")))]
    (clauth.token/find-valid-token token)))

(defn wrap-token-auth
  "Adds authentication info to the request data.
  :authenticated? says whether request was authenticated before.
  :principal holds informtion about authenticated entity"

  [handler]
  (fn [req]
    (let [token (find-token req)]
      (handler (assoc req
                      :principal (:subject token)
                      :authenticated? (not (nil? token)))))))

(defn wrap-internal-error [handler & {:keys [log error-response error-response-handler]}]
  (fn [req]
    (try (handler req)
         (catch Throwable t
           (if log (log t) (.printStackTrace t))
           {:status 500
            :headers {"Content-Type" "text/html"}
            :body (if error-response-handler
                    (error-response-handler req)
                    error-response)}))))

(defn wrap-secured [handler authorize-p unauthorized-response]
  (fn [req]
    (if (or (nil? authorize-p)
            (and (:authenticated? req) (authorize-p req)))
      (handler req)
      unauthorized-response)))

(defn wrap-logging [handler msg]
  (fn [req]
    (log/info msg (:session req))
    (handler req)))

(defn wrap-api-middleware
  ([routes]
   (wrap-api-middleware routes nil))
  ([routes authorize-p]
   (wrap-routes routes (fn [handler]
                         (-> handler
                             (wrap-secured authorize-p {:status 403})
                             (wrap-token-auth)
                             (wrap-defaults (assoc api-defaults
                                                   :cookies true
                                                   :session (session-store (get-in app-config [:cookie :secret]))
                                                   :security nil)))))))

(defn wrap-uri-middleware
  ([routes]
   (wrap-uri-middleware routes nil))
  ([routes authorize-p]
   (wrap-routes routes (fn [handler]
                         (-> handler
                             (wrap-secured authorize-p {:status 302 :headers {"Location" "/login"}})
                             (wrap-token-auth)
                             (wrap-defaults (assoc site-defaults
                                                   :session (session-store (get-in app-config [:cookie :secret])))))))))

(defn development-middlewares [handler]
  (log/info "Development middlewares applied")
  (-> handler
      (wrap-error-page)
      (wrap-exceptions {:app-namespaces ["cerber"]})))

(defn production-middlewares [handler]
  (log/info "Production middlewares applied")
  (-> handler
      (wrap-internal-error :log (fn [e] (log/error e)) :error-response "Something bad happened.")))

(defn wrap-middlewares [handler]
  (if (:is-dev? app-config)
    (development-middlewares handler)
    (production-middlewares handler)))
