(ns cerber.authenticator
  (:require [compojure.core        :refer [POST GET routes]]
            [clojure.string        :as str]
            [clojure.tools.logging :as log]
            [cerber.db.store       :as store]
            [cerber.http.response  :as response]
            [cheshire.core         :as json]
            [clauth.user           :as user]
            [clauth.token          :as token]
            [clj-http.client       :as client]
            [clj-oauth2.client     :as oauth2]))

(defn normalize-login
  "Normalizing login to required rules:
  - all lower-cased,
  - starts with letter followed by letters, digits, dot or underscore"

  [login]
  (re-find #"[a-z][a-z0-9\_\.]+" (-> login
                                     (str/lower-case)
                                     (str/replace #"[!@#\$\%\^\&\*\(\)\+\-\?<>]+" "")
                                     (str/replace " " "."))))

(defn generate-token
  "Generates OAuth token for given OAuth client wired with user's identifier."

  [client user]
  (token/create-token client user))

(defn with-authorities
  "Extends user data with authorities provided in details at key defined in :about-me config"

  [user details aboutme-config]
  (merge user {:authorities (details (get aboutme-config :authorities "authorities"))}))

(defn redirect-to-landing-page
  "Generates cookie with access token based on user's details and redirect user to configured lading page."

  [details config client req]
  (response/redirect-to (:landing-page config)
                        {:access_token (:token (generate-token client (select-keys details [:login :email :name :authorities])))}))

(defn request-access-token [config req]
  (try
    (if-let [response (oauth2/get-access-token config (:params req) req)]
      (:access-token response))
    (catch Exception e
      (log/error "Cannot receive access-token. " (.getMessage e)))))

(defn request-user-details
  "Returns user's data from OAuth provider if access-token was valid or nil otherwise."

  [url headers access-token]
  (when-not (empty? access-token)
    (when-let [response (client/get (str/replace-first url "{token}" access-token) {:headers headers})]
      (json/parse-string (:body response)))))

(defn login-handler [config req]
  (response/redirect-to
   (:uri (oauth2/make-auth-request config))))

(defn registration-handler [{aboutme :about-me} {{:keys [token login]} :params}]
  (when-let [details (request-user-details (:url aboutme) (:headers aboutme) token)]
    (if (store/fetch-user-by-login login)
      (response/user-exists)
      (with-authorities
        (user/register-user (store/bare-new-user {:login login
                                                  :email (details (get aboutme :email "email"))
                                                  :oid   (details (get aboutme :id    "id"))
                                                  :name  (details (get aboutme :name  "name"))
                                                  :active? true}))
        details aboutme))))


(defn auto-registration-handler
  "Basing on :auto-register? param of :authenticator config either redirects to registration page
  or delegates instantly registration process to registration-handler, which should end up with user registration."

  [details config client req]
  (if-not (:auto-register? config)
    (response/render-form "auth/register.html" details req)
    (when-let [registered (registration-handler config {:params details})]
      (redirect-to-landing-page registered config client req))))

(defn authentication-handler
  "Authenticates user through external OAuth provider (like facebook).
  Authenticated user is redirected to configured :landing-page, while unauthorized one (depending on :auto-register?) param
  is either redirected to registration form or gets registered automatically based on details returned by OAuth provider."

  [config client req]
  (if-let [token  (request-access-token config req)]
    (let [aboutme (:about-me config)
          details (request-user-details (:url aboutme) (:headers aboutme) token)]

      (if-let [user (store/fetch-user-by-oid (details (get aboutme :id "id")))]
        (redirect-to-landing-page  (with-authorities user details aboutme) config client req)
        (auto-registration-handler {:action "/oauth/register"
                                    :url    "/oauth/login"
                                    :email  (details (get aboutme :email "email"))
                                    :login  (normalize-login (details (get aboutme :name "name")))
                                    :token  token}
                                   config
                                   client
                                   req)))

    ;; something went wrong - let the user know.
    ;; usually it's enough just to log in again.

    (response/render-page "auth/exception.html")))

(defn init-routes
  "Initializes Authenticator routes"

  [client config]
  (routes
   (POST "/oauth/register" [] (partial registration-handler config))
   (GET  "/oauth/callback" [] (partial authentication-handler config client))
   (GET  "/oauth/login"    [] (partial login-handler config))))
