(ns cerber.http.response
  (:require [liberator.representation :refer [as-response ring-response]]
            [clauth.middleware        :as middleware]
            [clojure.string           :as str]
            [ring.util.response       :as response]
            [selmer.parser            :as selmer]))

(defn is-same?
  "Truthy if two maps contain same :login, falsy otherwise"

  [user principal]
  (when-let [login (:login user)]
    (= login (:login principal))))

(defn filter-sensitives
  "Filters out sensitive data from given map representing user data"

  [user]
  (dissoc user :email :oid :password :encrypt-password :confirmation_token :last_login :is_confirmation_sent))

(defn filtered-response
  "If response :status is 200 OK returns entire response content filtered out of sensitive data like
  email, auth id, password or tokens. Additionally if response goes to user different than
  the one described by content, roles are stripped as well for security reasons.
  In case of any other response status, original response gets returned unaltered."

  [response ctx]
  (as-response (if (= (:status ctx) 200)
                 (let [principal (get-in ctx [:request :principal])
                       filtered  (filter-sensitives response)]
                   (if (is-same? filtered principal)
                     filtered
                     (dissoc filtered :id :roles)))
                 response) ctx))

(defn render-form
  "Renders CSRF protected form based on given template and parameters"

  [template params req]
  (-> (selmer/render-file template (assoc params :csrf (middleware/csrf-token req)))
      (response/response)
      (response/header "Content-Type" "text/html")))

(defn render-page
  ([template]
   (render-page template {}))
  ([template params]
   (-> (selmer/render-file template params)
       (response/response)
       (response/header "Content-Type" "text/html")))
  ([template params req]
   (render-page template (assoc params
                                :user (filter-sensitives (:principal req))
                                :authenticated? (:authenticated? req)))))

(defn redirect-to
  "Redirects browser to given location"

  ([location]
   (response/redirect location))
  ([location session]
   (-> (redirect-to location)
       (assoc :session session))))

(defn user-exists
  "HTTP response with 409 Conflict"

  []
  (ring-response {:status 409 :body "User already exists"}))

(defn allowed-methods
  "HTTP response with Allow header set depending on user <-> principal comparison"

  [user principal]
  (ring-response {:headers
                  {"Allow"
                   (if-not user
                     "POST"
                     (if (is-same? principal user)
                       "GET,POST,PUT,DELETE"
                       "GET"))}}))
