(ns cerber.db.store
  (:require [clojure.tools.logging   :as log]
            [taoensso.carmine        :as car]
            [korma.db                :as korma]
            [clauth.user             :as user]
            [clauth.client           :as client]
            [clauth.token            :as token]
            [clauth.auth-code        :as auth-code]
            [clauth.store.redis      :as redis]
            [clauth.store            :refer [Store]]
            [cerber.db.schema.client :refer :all]
            [cerber.db.schema.user   :refer :all]))

(defprotocol ExtendedStore
  (fetch-by-oid [this t] "finds user by OAuth id"))

(defrecord SqlClientStore []
  Store
  (fetch [this t] (find-by-client-id t))
  (revoke! [this t] (remove-client t))
  (store! [this t item] (create-client item))
  (entries [this] (find-clients))
  (reset-store! [this]))

(defrecord SqlUserStore []
  Store
  (fetch [this t] (find-by-login t))
  (revoke! [this t] (remove-user t))
  (store! [this t item] (create-user item))
  (entries [this] (find-users))
  (reset-store! [this])

  ExtendedStore
  (fetch-by-oid [this t] (find-by-oid t)))

(defrecord RedisStore [namespace server-conn]
  Store
  (fetch [this t] (redis/wcar* (car/get (str namespace "/" t))))
  (revoke! [this t] (redis/wcar* (car/del (str namespace "/" t))))
  (store! [this t item]
    (redis/wcar* (car/set (str namespace "/" (t item)) item))
    (when-let [oid (:oid item)]
      (redis/wcar* (car/set (str "oid/" oid) (t item))))
    item)
  (entries [this] (redis/all-in-namespace namespace server-conn))
  (reset-store! [this] (redis/wcar* (car/flushdb)))

  ExtendedStore
  (fetch-by-oid [this t]
    (when-let [k (redis/wcar* (car/get (str "oid/" t)))]
      (.fetch this k))))

(defrecord MemoryStore [namespace store]
  Store
  (fetch [this t] (get @store (str namespace "/" t)))
  (revoke! [this t] (swap! store dissoc (str namespace "/" t)))
  (store! [this t item]
    (swap! store assoc (str namespace "/" (t item)) item)
    (when-let [oid (:oid item)]
      (swap! store assoc (str "oid/" oid) (t item)))
    item)
  (entries [this] (vals @store))
  (reset-store! [this] (swap! store empty))

  ExtendedStore
  (fetch-by-oid [this t]
    (when-let [k (get @store (str "oid/" t))]
      (.fetch this k))))

(defmulti create-ident-store :ident-store)
(defmulti create-token-store :token-store)

(defmethod create-ident-store :sql [config]
  (log/info "creating SQL backed store")
  (korma/defdb db (:db-spec config))

  (reset! client/client-store (SqlClientStore.))
  (reset! user/user-store (SqlUserStore.)))

(defmethod create-ident-store :in-memory [config]
  (log/info "creating in-memory backed store")

  (reset! client/client-store (MemoryStore. "clients" (atom {})))
  (reset! user/user-store (MemoryStore. "users" (atom {}))))

(defmethod create-ident-store :redis [config]
  (log/info "creating Redis backed store")

  (reset! client/client-store (RedisStore. "clients" (:db-spec config)))
  (reset! user/user-store (RedisStore. "users" (:db-spec config))))

(defmethod create-token-store :redis [config]
  (reset! token/token-store (redis/create-redis-store "tokens" (:db-spec config)))
  (reset! auth-code/auth-code-store (redis/create-redis-store "auth-codes" (:db-spec config))))

(defmethod create-token-store :in-memory [config]
  (reset! token/token-store (MemoryStore. "tokens" (atom {})))
  (reset! auth-code/auth-code-store (MemoryStore. "auth-codes" (atom {}))))

(defn fetch-user-by-oid
  "Returns user's core data basing on OAuth id"

  [oid]
  (fetch-by-oid @user/user-store oid))

(defn fetch-user-by-login
  "Returns user's core data through underlaying OAuth store"

  [login]
  (clauth.user/fetch-user login))

(defn bare-new-user
  "Returns bare user data. Used primarily to proceed with registration of newly created user."

  [{:keys [login name email password oid authorities active?]}]
  {:login login
   :name  name
   :email email
   :encrypt-password password
   :confirmation_token (.toString (BigInteger. 130 (java.security.SecureRandom.)))
   :oid oid
   :authorities authorities
   :is_active (boolean (or active? oid))
   :is_confirmed (boolean oid)
   :is_confirmation_sent false})
